#[cfg(feature = "radix")]
mod util;

use core::fmt::Debug;
use core::str::{from_utf8_unchecked, FromStr};
use lexical_util::constants::FormattedSize;
#[cfg(feature = "radix")]
use lexical_util::constants::BUFFER_SIZE;
#[cfg(feature = "format")]
use lexical_util::format::NumberFormatBuilder;
use lexical_util::format::STANDARD;
use lexical_write_integer::{Options, ToLexical, ToLexicalWithOptions};
use proptest::prelude::*;
use quickcheck::quickcheck;
#[cfg(feature = "radix")]
use util::from_radix;

trait Roundtrip: ToLexical + ToLexicalWithOptions + FromStr {
    fn from_str_radix(src: &str, radix: u32) -> Result<Self, std::num::ParseIntError>;
}

macro_rules! roundtrip_impl {
    ($($t:ty)*) => ($(
        impl Roundtrip for $t {
            fn from_str_radix(src: &str, radix: u32) -> Result<Self, std::num::ParseIntError> {
                <$t>::from_str_radix(src, radix)
            }
        }
    )*);
}

roundtrip_impl! { u8 u16 u32 u64 u128 usize i8 i16 i32 i64 i128 isize }

#[test]
#[cfg(feature = "format")]
fn mandatory_sign_test() {
    let mut buffer = [b'\x00'; 16];
    let options = Options::new();
    const FORMAT: u128 = NumberFormatBuilder::new().required_mantissa_sign(true).build();
    assert_eq!(b"+0", 0i8.to_lexical_with_options::<{ FORMAT }>(&mut buffer, &options));
    assert_eq!(b"-1", (-1i8).to_lexical_with_options::<{ FORMAT }>(&mut buffer, &options));
    assert_eq!(b"+1", 1i8.to_lexical_with_options::<{ FORMAT }>(&mut buffer, &options));
}

#[test]
fn u8_test() {
    let mut buffer = [b'\x00'; 16];
    assert_eq!(b"0", 0u8.to_lexical(&mut buffer));
    assert_eq!(b"1", 1u8.to_lexical(&mut buffer));
    assert_eq!(b"5", 5u8.to_lexical(&mut buffer));
    assert_eq!(b"127", 127u8.to_lexical(&mut buffer));
    assert_eq!(b"128", 128u8.to_lexical(&mut buffer));
    assert_eq!(b"255", 255u8.to_lexical(&mut buffer));
    assert_eq!(b"255", (-1i8 as u8).to_lexical(&mut buffer));
}

#[test]
fn i8_test() {
    let mut buffer = [b'\x00'; 16];
    assert_eq!(b"0", 0i8.to_lexical(&mut buffer));
    assert_eq!(b"1", 1i8.to_lexical(&mut buffer));
    assert_eq!(b"5", 5i8.to_lexical(&mut buffer));
    assert_eq!(b"127", 127i8.to_lexical(&mut buffer));
    assert_eq!(b"-128", (128u8 as i8).to_lexical(&mut buffer));
    assert_eq!(b"-1", (255u8 as i8).to_lexical(&mut buffer));
    assert_eq!(b"-1", (-1i8).to_lexical(&mut buffer));
}

#[test]
fn u16_test() {
    let mut buffer = [b'\x00'; 16];
    assert_eq!(b"0", 0u16.to_lexical(&mut buffer));
    assert_eq!(b"1", 1u16.to_lexical(&mut buffer));
    assert_eq!(b"5", 5u16.to_lexical(&mut buffer));
    assert_eq!(b"32767", 32767u16.to_lexical(&mut buffer));
    assert_eq!(b"32768", 32768u16.to_lexical(&mut buffer));
    assert_eq!(b"65535", 65535u16.to_lexical(&mut buffer));
    assert_eq!(b"65535", (-1i16 as u16).to_lexical(&mut buffer));
}

#[test]
fn i16_test() {
    let mut buffer = [b'\x00'; 16];
    assert_eq!(b"0", 0i16.to_lexical(&mut buffer));
    assert_eq!(b"1", 1i16.to_lexical(&mut buffer));
    assert_eq!(b"5", 5i16.to_lexical(&mut buffer));
    assert_eq!(b"32767", 32767i16.to_lexical(&mut buffer));
    assert_eq!(b"-32768", (32768u16 as i16).to_lexical(&mut buffer));
    assert_eq!(b"-1", (65535u16 as i16).to_lexical(&mut buffer));
    assert_eq!(b"-1", (-1i16).to_lexical(&mut buffer));
}

#[test]
fn u32_test() {
    let mut buffer = [b'\x00'; 16];
    assert_eq!(b"0", 0u32.to_lexical(&mut buffer));
    assert_eq!(b"1", 1u32.to_lexical(&mut buffer));
    assert_eq!(b"5", 5u32.to_lexical(&mut buffer));
    assert_eq!(b"2147483647", 2147483647u32.to_lexical(&mut buffer));
    assert_eq!(b"2147483648", 2147483648u32.to_lexical(&mut buffer));
    assert_eq!(b"4294967295", 4294967295u32.to_lexical(&mut buffer));
    assert_eq!(b"4294967295", (-1i32 as u32).to_lexical(&mut buffer));
}

#[test]
fn i32_test() {
    let mut buffer = [b'\x00'; 16];
    assert_eq!(b"0", 0i32.to_lexical(&mut buffer));
    assert_eq!(b"1", 1i32.to_lexical(&mut buffer));
    assert_eq!(b"5", 5i32.to_lexical(&mut buffer));
    assert_eq!(b"2147483647", 2147483647i32.to_lexical(&mut buffer));
    assert_eq!(b"-2147483648", (2147483648u32 as i32).to_lexical(&mut buffer));
    assert_eq!(b"-1", (4294967295u32 as i32).to_lexical(&mut buffer));
    assert_eq!(b"-1", (-1i32).to_lexical(&mut buffer));
}

#[test]
fn u64_test() {
    let mut buffer = [b'\x00'; 32];
    assert_eq!(b"0", 0u64.to_lexical(&mut buffer));
    assert_eq!(b"1", 1u64.to_lexical(&mut buffer));
    assert_eq!(b"5", 5u64.to_lexical(&mut buffer));
    assert_eq!(b"9223372036854775807", 9223372036854775807u64.to_lexical(&mut buffer));
    assert_eq!(b"9223372036854775808", 9223372036854775808u64.to_lexical(&mut buffer));
    assert_eq!(b"18446744073709551615", 18446744073709551615u64.to_lexical(&mut buffer));
    assert_eq!(b"18446744073709551615", (-1i64 as u64).to_lexical(&mut buffer));
}

#[test]
fn i64_test() {
    let mut buffer = [b'\x00'; 32];
    assert_eq!(b"0", 0i64.to_lexical(&mut buffer));
    assert_eq!(b"1", 1i64.to_lexical(&mut buffer));
    assert_eq!(b"5", 5i64.to_lexical(&mut buffer));
    assert_eq!(b"9223372036854775807", 9223372036854775807i64.to_lexical(&mut buffer));
    assert_eq!(b"-9223372036854775808", (9223372036854775808u64 as i64).to_lexical(&mut buffer));
    assert_eq!(b"-1", (18446744073709551615u64 as i64).to_lexical(&mut buffer));
    assert_eq!(b"-1", (-1i64).to_lexical(&mut buffer));
}

#[test]
fn u128_test() {
    let mut buffer = [b'\x00'; 48];
    assert_eq!(b"0", 0u128.to_lexical(&mut buffer));
    assert_eq!(b"1", 1u128.to_lexical(&mut buffer));
    assert_eq!(b"5", 5u128.to_lexical(&mut buffer));
    assert_eq!(
        &b"170141183460469231731687303715884105727"[..],
        170141183460469231731687303715884105727u128.to_lexical(&mut buffer)
    );
    assert_eq!(
        &b"170141183460469231731687303715884105728"[..],
        170141183460469231731687303715884105728u128.to_lexical(&mut buffer)
    );
    assert_eq!(
        &b"340282366920938463463374607431768211455"[..],
        340282366920938463463374607431768211455u128.to_lexical(&mut buffer)
    );
    assert_eq!(
        &b"340282366920938463463374607431768211455"[..],
        (-1i128 as u128).to_lexical(&mut buffer)
    );
}

#[test]
fn i128_test() {
    let mut buffer = [b'\x00'; 48];
    assert_eq!(b"0", 0i128.to_lexical(&mut buffer));
    assert_eq!(b"1", 1i128.to_lexical(&mut buffer));
    assert_eq!(b"5", 5i128.to_lexical(&mut buffer));
    assert_eq!(
        &b"170141183460469231731687303715884105727"[..],
        170141183460469231731687303715884105727i128.to_lexical(&mut buffer)
    );
    assert_eq!(
        &b"-170141183460469231731687303715884105728"[..],
        (170141183460469231731687303715884105728u128 as i128).to_lexical(&mut buffer)
    );
    assert_eq!(
        b"-1",
        (340282366920938463463374607431768211455u128 as i128).to_lexical(&mut buffer)
    );
    assert_eq!(b"-1", (-1i128).to_lexical(&mut buffer));
}

#[test]
#[cfg(feature = "radix")]
fn proptest_failures_radix() {
    const FORMAT: u128 = from_radix(12);
    let mut buffer = [b'\x00'; BUFFER_SIZE];
    let options = Options::new();
    assert_eq!(b"A8", 128u8.to_lexical_with_options::<{ FORMAT }>(&mut buffer, &options));
}

#[test]
fn options_test() {
    let mut buffer = [b'\x00'; 48];
    let options = Options::new();
    assert_eq!(b"0", 0u8.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0u16.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0u32.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0u64.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0u128.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0i8.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0i16.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0i32.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0i64.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
    assert_eq!(b"0", 0i128.to_lexical_with_options::<{ STANDARD }>(&mut buffer, &options));
}

#[test]
#[cfg(feature = "radix")]
fn options_radix_test() {
    const FORMAT: u128 = from_radix(12);
    let mut buffer = [b'\x00'; 48];
    let options = Options::new();
    assert_eq!(b"A8", 128u8.to_lexical_with_options::<{ FORMAT }>(&mut buffer, &options));
}

fn roundtrip<T>(x: T) -> T
where
    T: Roundtrip,
    <T as FromStr>::Err: Debug,
{
    let mut buffer = [b'\x00'; 48];
    let bytes = x.to_lexical(&mut buffer);
    let string = unsafe { from_utf8_unchecked(bytes) };
    string.parse::<T>().unwrap()
}

#[cfg(feature = "radix")]
fn roundtrip_radix<T>(x: T, radix: u32) -> T
where
    T: Roundtrip,
    <T as FromStr>::Err: Debug,
{
    let mut buffer = [b'\x00'; BUFFER_SIZE];
    let options = T::Options::default();
    // Trick it into assuming we have a valid radix.
    let bytes = match radix {
        2 => x.to_lexical_with_options::<{ from_radix(2) }>(&mut buffer, &options),
        3 => x.to_lexical_with_options::<{ from_radix(3) }>(&mut buffer, &options),
        4 => x.to_lexical_with_options::<{ from_radix(4) }>(&mut buffer, &options),
        5 => x.to_lexical_with_options::<{ from_radix(5) }>(&mut buffer, &options),
        6 => x.to_lexical_with_options::<{ from_radix(6) }>(&mut buffer, &options),
        7 => x.to_lexical_with_options::<{ from_radix(7) }>(&mut buffer, &options),
        8 => x.to_lexical_with_options::<{ from_radix(8) }>(&mut buffer, &options),
        9 => x.to_lexical_with_options::<{ from_radix(9) }>(&mut buffer, &options),
        10 => x.to_lexical_with_options::<{ from_radix(10) }>(&mut buffer, &options),
        11 => x.to_lexical_with_options::<{ from_radix(11) }>(&mut buffer, &options),
        12 => x.to_lexical_with_options::<{ from_radix(12) }>(&mut buffer, &options),
        13 => x.to_lexical_with_options::<{ from_radix(13) }>(&mut buffer, &options),
        14 => x.to_lexical_with_options::<{ from_radix(14) }>(&mut buffer, &options),
        15 => x.to_lexical_with_options::<{ from_radix(15) }>(&mut buffer, &options),
        16 => x.to_lexical_with_options::<{ from_radix(16) }>(&mut buffer, &options),
        17 => x.to_lexical_with_options::<{ from_radix(17) }>(&mut buffer, &options),
        18 => x.to_lexical_with_options::<{ from_radix(18) }>(&mut buffer, &options),
        19 => x.to_lexical_with_options::<{ from_radix(19) }>(&mut buffer, &options),
        20 => x.to_lexical_with_options::<{ from_radix(20) }>(&mut buffer, &options),
        21 => x.to_lexical_with_options::<{ from_radix(21) }>(&mut buffer, &options),
        22 => x.to_lexical_with_options::<{ from_radix(22) }>(&mut buffer, &options),
        23 => x.to_lexical_with_options::<{ from_radix(23) }>(&mut buffer, &options),
        24 => x.to_lexical_with_options::<{ from_radix(24) }>(&mut buffer, &options),
        25 => x.to_lexical_with_options::<{ from_radix(25) }>(&mut buffer, &options),
        26 => x.to_lexical_with_options::<{ from_radix(26) }>(&mut buffer, &options),
        27 => x.to_lexical_with_options::<{ from_radix(27) }>(&mut buffer, &options),
        28 => x.to_lexical_with_options::<{ from_radix(28) }>(&mut buffer, &options),
        29 => x.to_lexical_with_options::<{ from_radix(29) }>(&mut buffer, &options),
        30 => x.to_lexical_with_options::<{ from_radix(30) }>(&mut buffer, &options),
        31 => x.to_lexical_with_options::<{ from_radix(31) }>(&mut buffer, &options),
        32 => x.to_lexical_with_options::<{ from_radix(32) }>(&mut buffer, &options),
        33 => x.to_lexical_with_options::<{ from_radix(33) }>(&mut buffer, &options),
        34 => x.to_lexical_with_options::<{ from_radix(34) }>(&mut buffer, &options),
        35 => x.to_lexical_with_options::<{ from_radix(35) }>(&mut buffer, &options),
        36 => x.to_lexical_with_options::<{ from_radix(36) }>(&mut buffer, &options),
        _ => unreachable!(),
    };
    let string = unsafe { from_utf8_unchecked(bytes) };
    T::from_str_radix(string, radix).unwrap()
}

#[test]
fn u8_pow2_test() {
    let values: &[u8] =
        &[0, 1, 2, 3, 4, 5, 7, 8, 9, 15, 16, 17, 31, 32, 33, 63, 64, 65, 127, 128, 129, 255];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u8_pow10_test() {
    let values: &[u8] = &[0, 1, 5, 9, 10, 11, 15, 99, 100, 101, 105];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u16_pow2_test() {
    let values: &[u16] = &[
        0, 1, 2, 3, 4, 5, 7, 8, 9, 15, 16, 17, 31, 32, 33, 63, 64, 65, 127, 128, 129, 255, 256,
        257, 511, 512, 513, 1023, 1024, 1025, 2047, 2048, 2049, 4095, 4096, 4097, 8191, 8192, 8193,
        16383, 16384, 16385, 32767, 32768, 32769, 65535,
    ];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u16_pow10_test() {
    let values: &[u16] = &[
        0, 1, 5, 9, 10, 11, 15, 99, 100, 101, 105, 999, 1000, 1001, 1005, 9999, 10000, 10001, 10005,
    ];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u32_pow2_test() {
    let values: &[u32] = &[
        0, 1, 2, 3, 4, 5, 7, 8, 9, 15, 16, 17, 31, 32, 33, 63, 64, 65, 127, 128, 129, 255, 256,
        257, 511, 512, 513, 1023, 1024, 1025, 2047, 2048, 2049, 4095, 4096, 4097, 8191, 8192, 8193,
        16383, 16384, 16385, 32767, 32768, 32769, 65535, 65536, 65537, 131071, 131072, 131073,
        262143, 262144, 262145, 524287, 524288, 524289, 1048575, 1048576, 1048577, 2097151,
        2097152, 2097153, 4194303, 4194304, 4194305, 8388607, 8388608, 8388609, 16777215, 16777216,
        16777217, 33554431, 33554432, 33554433, 67108863, 67108864, 67108865, 134217727, 134217728,
        134217729, 268435455, 268435456, 268435457, 536870911, 536870912, 536870913, 1073741823,
        1073741824, 1073741825, 2147483647, 2147483648, 2147483649, 4294967295,
    ];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u32_pow10_test() {
    let values: &[u32] = &[
        0, 1, 5, 9, 10, 11, 15, 99, 100, 101, 105, 999, 1000, 1001, 1005, 9999, 10000, 10001,
        10005, 99999, 100000, 100001, 100005, 999999, 1000000, 1000001, 1000005, 9999999, 10000000,
        10000001, 10000005, 99999999, 100000000, 100000001, 100000005, 999999999, 1000000000,
        1000000001, 1000000005,
    ];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u64_pow2_test() {
    let values: &[u64] = &[
        0,
        1,
        2,
        3,
        4,
        5,
        7,
        8,
        9,
        15,
        16,
        17,
        31,
        32,
        33,
        63,
        64,
        65,
        127,
        128,
        129,
        255,
        256,
        257,
        511,
        512,
        513,
        1023,
        1024,
        1025,
        2047,
        2048,
        2049,
        4095,
        4096,
        4097,
        8191,
        8192,
        8193,
        16383,
        16384,
        16385,
        32767,
        32768,
        32769,
        65535,
        65536,
        65537,
        131071,
        131072,
        131073,
        262143,
        262144,
        262145,
        524287,
        524288,
        524289,
        1048575,
        1048576,
        1048577,
        2097151,
        2097152,
        2097153,
        4194303,
        4194304,
        4194305,
        8388607,
        8388608,
        8388609,
        16777215,
        16777216,
        16777217,
        33554431,
        33554432,
        33554433,
        67108863,
        67108864,
        67108865,
        134217727,
        134217728,
        134217729,
        268435455,
        268435456,
        268435457,
        536870911,
        536870912,
        536870913,
        1073741823,
        1073741824,
        1073741825,
        2147483647,
        2147483648,
        2147483649,
        4294967295,
        4294967296,
        4294967297,
        8589934591,
        8589934592,
        8589934593,
        17179869183,
        17179869184,
        17179869185,
        34359738367,
        34359738368,
        34359738369,
        68719476735,
        68719476736,
        68719476737,
        137438953471,
        137438953472,
        137438953473,
        274877906943,
        274877906944,
        274877906945,
        549755813887,
        549755813888,
        549755813889,
        1099511627775,
        1099511627776,
        1099511627777,
        2199023255551,
        2199023255552,
        2199023255553,
        4398046511103,
        4398046511104,
        4398046511105,
        8796093022207,
        8796093022208,
        8796093022209,
        17592186044415,
        17592186044416,
        17592186044417,
        35184372088831,
        35184372088832,
        35184372088833,
        70368744177663,
        70368744177664,
        70368744177665,
        140737488355327,
        140737488355328,
        140737488355329,
        281474976710655,
        281474976710656,
        281474976710657,
        562949953421311,
        562949953421312,
        562949953421313,
        1125899906842623,
        1125899906842624,
        1125899906842625,
        2251799813685247,
        2251799813685248,
        2251799813685249,
        4503599627370495,
        4503599627370496,
        4503599627370497,
        9007199254740991,
        9007199254740992,
        9007199254740993,
        18014398509481983,
        18014398509481984,
        18014398509481985,
        36028797018963967,
        36028797018963968,
        36028797018963969,
        72057594037927935,
        72057594037927936,
        72057594037927937,
        144115188075855871,
        144115188075855872,
        144115188075855873,
        288230376151711743,
        288230376151711744,
        288230376151711745,
        576460752303423487,
        576460752303423488,
        576460752303423489,
        1152921504606846975,
        1152921504606846976,
        1152921504606846977,
        2305843009213693951,
        2305843009213693952,
        2305843009213693953,
        4611686018427387903,
        4611686018427387904,
        4611686018427387905,
        9223372036854775807,
        9223372036854775808,
        9223372036854775809,
        18446744073709551615,
    ];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u64_pow10_test() {
    let values: &[u64] = &[
        0,
        1,
        5,
        9,
        10,
        11,
        15,
        99,
        100,
        101,
        105,
        999,
        1000,
        1001,
        1005,
        9999,
        10000,
        10001,
        10005,
        99999,
        100000,
        100001,
        100005,
        999999,
        1000000,
        1000001,
        1000005,
        9999999,
        10000000,
        10000001,
        10000005,
        99999999,
        100000000,
        100000001,
        100000005,
        999999999,
        1000000000,
        1000000001,
        1000000005,
        9999999999,
        10000000000,
        10000000001,
        10000000005,
        99999999999,
        100000000000,
        100000000001,
        100000000005,
        999999999999,
        1000000000000,
        1000000000001,
        1000000000005,
        9999999999999,
        10000000000000,
        10000000000001,
        10000000000005,
        99999999999999,
        100000000000000,
        100000000000001,
        100000000000005,
        999999999999999,
        1000000000000000,
        1000000000000001,
        1000000000000005,
        9999999999999999,
        10000000000000000,
        10000000000000001,
        10000000000000005,
        99999999999999999,
        100000000000000000,
        100000000000000001,
        100000000000000005,
        999999999999999999,
        1000000000000000000,
        1000000000000000001,
        1000000000000000005,
    ];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u128_pow2_test() {
    let values: &[u128] = &[
        0,
        1,
        2,
        3,
        4,
        5,
        7,
        8,
        9,
        15,
        16,
        17,
        31,
        32,
        33,
        63,
        64,
        65,
        127,
        128,
        129,
        255,
        256,
        257,
        511,
        512,
        513,
        1023,
        1024,
        1025,
        2047,
        2048,
        2049,
        4095,
        4096,
        4097,
        8191,
        8192,
        8193,
        16383,
        16384,
        16385,
        32767,
        32768,
        32769,
        65535,
        65536,
        65537,
        131071,
        131072,
        131073,
        262143,
        262144,
        262145,
        524287,
        524288,
        524289,
        1048575,
        1048576,
        1048577,
        2097151,
        2097152,
        2097153,
        4194303,
        4194304,
        4194305,
        8388607,
        8388608,
        8388609,
        16777215,
        16777216,
        16777217,
        33554431,
        33554432,
        33554433,
        67108863,
        67108864,
        67108865,
        134217727,
        134217728,
        134217729,
        268435455,
        268435456,
        268435457,
        536870911,
        536870912,
        536870913,
        1073741823,
        1073741824,
        1073741825,
        2147483647,
        2147483648,
        2147483649,
        4294967295,
        4294967296,
        4294967297,
        8589934591,
        8589934592,
        8589934593,
        17179869183,
        17179869184,
        17179869185,
        34359738367,
        34359738368,
        34359738369,
        68719476735,
        68719476736,
        68719476737,
        137438953471,
        137438953472,
        137438953473,
        274877906943,
        274877906944,
        274877906945,
        549755813887,
        549755813888,
        549755813889,
        1099511627775,
        1099511627776,
        1099511627777,
        2199023255551,
        2199023255552,
        2199023255553,
        4398046511103,
        4398046511104,
        4398046511105,
        8796093022207,
        8796093022208,
        8796093022209,
        17592186044415,
        17592186044416,
        17592186044417,
        35184372088831,
        35184372088832,
        35184372088833,
        70368744177663,
        70368744177664,
        70368744177665,
        140737488355327,
        140737488355328,
        140737488355329,
        281474976710655,
        281474976710656,
        281474976710657,
        562949953421311,
        562949953421312,
        562949953421313,
        1125899906842623,
        1125899906842624,
        1125899906842625,
        2251799813685247,
        2251799813685248,
        2251799813685249,
        4503599627370495,
        4503599627370496,
        4503599627370497,
        9007199254740991,
        9007199254740992,
        9007199254740993,
        18014398509481983,
        18014398509481984,
        18014398509481985,
        36028797018963967,
        36028797018963968,
        36028797018963969,
        72057594037927935,
        72057594037927936,
        72057594037927937,
        144115188075855871,
        144115188075855872,
        144115188075855873,
        288230376151711743,
        288230376151711744,
        288230376151711745,
        576460752303423487,
        576460752303423488,
        576460752303423489,
        1152921504606846975,
        1152921504606846976,
        1152921504606846977,
        2305843009213693951,
        2305843009213693952,
        2305843009213693953,
        4611686018427387903,
        4611686018427387904,
        4611686018427387905,
        9223372036854775807,
        9223372036854775808,
        9223372036854775809,
        18446744073709551615,
        18446744073709551616,
        18446744073709551617,
        36893488147419103231,
        36893488147419103232,
        36893488147419103233,
        73786976294838206463,
        73786976294838206464,
        73786976294838206465,
        147573952589676412927,
        147573952589676412928,
        147573952589676412929,
        295147905179352825855,
        295147905179352825856,
        295147905179352825857,
        590295810358705651711,
        590295810358705651712,
        590295810358705651713,
        1180591620717411303423,
        1180591620717411303424,
        1180591620717411303425,
        2361183241434822606847,
        2361183241434822606848,
        2361183241434822606849,
        4722366482869645213695,
        4722366482869645213696,
        4722366482869645213697,
        9444732965739290427391,
        9444732965739290427392,
        9444732965739290427393,
        18889465931478580854783,
        18889465931478580854784,
        18889465931478580854785,
        37778931862957161709567,
        37778931862957161709568,
        37778931862957161709569,
        75557863725914323419135,
        75557863725914323419136,
        75557863725914323419137,
        151115727451828646838271,
        151115727451828646838272,
        151115727451828646838273,
        302231454903657293676543,
        302231454903657293676544,
        302231454903657293676545,
        604462909807314587353087,
        604462909807314587353088,
        604462909807314587353089,
        1208925819614629174706175,
        1208925819614629174706176,
        1208925819614629174706177,
        2417851639229258349412351,
        2417851639229258349412352,
        2417851639229258349412353,
        4835703278458516698824703,
        4835703278458516698824704,
        4835703278458516698824705,
        9671406556917033397649407,
        9671406556917033397649408,
        9671406556917033397649409,
        19342813113834066795298815,
        19342813113834066795298816,
        19342813113834066795298817,
        38685626227668133590597631,
        38685626227668133590597632,
        38685626227668133590597633,
        77371252455336267181195263,
        77371252455336267181195264,
        77371252455336267181195265,
        154742504910672534362390527,
        154742504910672534362390528,
        154742504910672534362390529,
        309485009821345068724781055,
        309485009821345068724781056,
        309485009821345068724781057,
        618970019642690137449562111,
        618970019642690137449562112,
        618970019642690137449562113,
        1237940039285380274899124223,
        1237940039285380274899124224,
        1237940039285380274899124225,
        2475880078570760549798248447,
        2475880078570760549798248448,
        2475880078570760549798248449,
        4951760157141521099596496895,
        4951760157141521099596496896,
        4951760157141521099596496897,
        9903520314283042199192993791,
        9903520314283042199192993792,
        9903520314283042199192993793,
        19807040628566084398385987583,
        19807040628566084398385987584,
        19807040628566084398385987585,
        39614081257132168796771975167,
        39614081257132168796771975168,
        39614081257132168796771975169,
        79228162514264337593543950335,
        79228162514264337593543950336,
        79228162514264337593543950337,
        158456325028528675187087900671,
        158456325028528675187087900672,
        158456325028528675187087900673,
        316912650057057350374175801343,
        316912650057057350374175801344,
        316912650057057350374175801345,
        633825300114114700748351602687,
        633825300114114700748351602688,
        633825300114114700748351602689,
        1267650600228229401496703205375,
        1267650600228229401496703205376,
        1267650600228229401496703205377,
        2535301200456458802993406410751,
        2535301200456458802993406410752,
        2535301200456458802993406410753,
        5070602400912917605986812821503,
        5070602400912917605986812821504,
        5070602400912917605986812821505,
        10141204801825835211973625643007,
        10141204801825835211973625643008,
        10141204801825835211973625643009,
        20282409603651670423947251286015,
        20282409603651670423947251286016,
        20282409603651670423947251286017,
        40564819207303340847894502572031,
        40564819207303340847894502572032,
        40564819207303340847894502572033,
        81129638414606681695789005144063,
        81129638414606681695789005144064,
        81129638414606681695789005144065,
        162259276829213363391578010288127,
        162259276829213363391578010288128,
        162259276829213363391578010288129,
        324518553658426726783156020576255,
        324518553658426726783156020576256,
        324518553658426726783156020576257,
        649037107316853453566312041152511,
        649037107316853453566312041152512,
        649037107316853453566312041152513,
        1298074214633706907132624082305023,
        1298074214633706907132624082305024,
        1298074214633706907132624082305025,
        2596148429267413814265248164610047,
        2596148429267413814265248164610048,
        2596148429267413814265248164610049,
        5192296858534827628530496329220095,
        5192296858534827628530496329220096,
        5192296858534827628530496329220097,
        10384593717069655257060992658440191,
        10384593717069655257060992658440192,
        10384593717069655257060992658440193,
        20769187434139310514121985316880383,
        20769187434139310514121985316880384,
        20769187434139310514121985316880385,
        41538374868278621028243970633760767,
        41538374868278621028243970633760768,
        41538374868278621028243970633760769,
        83076749736557242056487941267521535,
        83076749736557242056487941267521536,
        83076749736557242056487941267521537,
        166153499473114484112975882535043071,
        166153499473114484112975882535043072,
        166153499473114484112975882535043073,
        332306998946228968225951765070086143,
        332306998946228968225951765070086144,
        332306998946228968225951765070086145,
        664613997892457936451903530140172287,
        664613997892457936451903530140172288,
        664613997892457936451903530140172289,
        1329227995784915872903807060280344575,
        1329227995784915872903807060280344576,
        1329227995784915872903807060280344577,
        2658455991569831745807614120560689151,
        2658455991569831745807614120560689152,
        2658455991569831745807614120560689153,
        5316911983139663491615228241121378303,
        5316911983139663491615228241121378304,
        5316911983139663491615228241121378305,
        10633823966279326983230456482242756607,
        10633823966279326983230456482242756608,
        10633823966279326983230456482242756609,
        21267647932558653966460912964485513215,
        21267647932558653966460912964485513216,
        21267647932558653966460912964485513217,
        42535295865117307932921825928971026431,
        42535295865117307932921825928971026432,
        42535295865117307932921825928971026433,
        85070591730234615865843651857942052863,
        85070591730234615865843651857942052864,
        85070591730234615865843651857942052865,
        170141183460469231731687303715884105727,
        170141183460469231731687303715884105728,
        170141183460469231731687303715884105729,
        340282366920938463463374607431768211455,
    ];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

#[test]
fn u128_pow10_test() {
    let values: &[u128] = &[
        0,
        1,
        5,
        9,
        10,
        11,
        15,
        99,
        100,
        101,
        105,
        999,
        1000,
        1001,
        1005,
        9999,
        10000,
        10001,
        10005,
        99999,
        100000,
        100001,
        100005,
        999999,
        1000000,
        1000001,
        1000005,
        9999999,
        10000000,
        10000001,
        10000005,
        99999999,
        100000000,
        100000001,
        100000005,
        999999999,
        1000000000,
        1000000001,
        1000000005,
        9999999999,
        10000000000,
        10000000001,
        10000000005,
        99999999999,
        100000000000,
        100000000001,
        100000000005,
        999999999999,
        1000000000000,
        1000000000001,
        1000000000005,
        9999999999999,
        10000000000000,
        10000000000001,
        10000000000005,
        99999999999999,
        100000000000000,
        100000000000001,
        100000000000005,
        999999999999999,
        1000000000000000,
        1000000000000001,
        1000000000000005,
        9999999999999999,
        10000000000000000,
        10000000000000001,
        10000000000000005,
        99999999999999999,
        100000000000000000,
        100000000000000001,
        100000000000000005,
        999999999999999999,
        1000000000000000000,
        1000000000000000001,
        1000000000000000005,
        9999999999999999999,
        10000000000000000000,
        10000000000000000001,
        10000000000000000005,
        99999999999999999999,
        100000000000000000000,
        100000000000000000001,
        100000000000000000005,
        999999999999999999999,
        1000000000000000000000,
        1000000000000000000001,
        1000000000000000000005,
        9999999999999999999999,
        10000000000000000000000,
        10000000000000000000001,
        10000000000000000000005,
        99999999999999999999999,
        100000000000000000000000,
        100000000000000000000001,
        100000000000000000000005,
        999999999999999999999999,
        1000000000000000000000000,
        1000000000000000000000001,
        1000000000000000000000005,
        9999999999999999999999999,
        10000000000000000000000000,
        10000000000000000000000001,
        10000000000000000000000005,
        99999999999999999999999999,
        100000000000000000000000000,
        100000000000000000000000001,
        100000000000000000000000005,
        999999999999999999999999999,
        1000000000000000000000000000,
        1000000000000000000000000001,
        1000000000000000000000000005,
        9999999999999999999999999999,
        10000000000000000000000000000,
        10000000000000000000000000001,
        10000000000000000000000000005,
        99999999999999999999999999999,
        100000000000000000000000000000,
        100000000000000000000000000001,
        100000000000000000000000000005,
        999999999999999999999999999999,
        1000000000000000000000000000000,
        1000000000000000000000000000001,
        1000000000000000000000000000005,
        9999999999999999999999999999999,
        10000000000000000000000000000000,
        10000000000000000000000000000001,
        10000000000000000000000000000005,
        99999999999999999999999999999999,
        100000000000000000000000000000000,
        100000000000000000000000000000001,
        100000000000000000000000000000005,
        999999999999999999999999999999999,
        1000000000000000000000000000000000,
        1000000000000000000000000000000001,
        1000000000000000000000000000000005,
        9999999999999999999999999999999999,
        10000000000000000000000000000000000,
        10000000000000000000000000000000001,
        10000000000000000000000000000000005,
        99999999999999999999999999999999999,
        100000000000000000000000000000000000,
        100000000000000000000000000000000001,
        100000000000000000000000000000000005,
        999999999999999999999999999999999999,
        1000000000000000000000000000000000000,
        1000000000000000000000000000000000001,
        1000000000000000000000000000000000005,
        9999999999999999999999999999999999999,
        10000000000000000000000000000000000000,
        10000000000000000000000000000000000001,
        10000000000000000000000000000000000005,
        99999999999999999999999999999999999999,
        100000000000000000000000000000000000000,
        100000000000000000000000000000000000001,
        100000000000000000000000000000000000005,
    ];
    for &i in values.iter() {
        assert_eq!(i, roundtrip(i));
    }
}

quickcheck! {
    #[cfg_attr(miri, ignore)]
    fn u8_quickcheck(i: u8) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn u16_quickcheck(i: u16) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn u32_quickcheck(i: u32) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn u64_quickcheck(i: u64) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn u128_quickcheck(i: u128) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn usize_quickcheck(i: usize) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn i8_quickcheck(i: i8) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn i16_quickcheck(i: i16) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn i32_quickcheck(i: i32) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn i64_quickcheck(i: i64) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn i128_quickcheck(i: i128) -> bool {
        i == roundtrip(i)
    }

    #[cfg_attr(miri, ignore)]
    fn isize_quickcheck(i: isize) -> bool {
        i == roundtrip(i)
    }
}

proptest! {
    #[test]
    #[cfg_attr(miri, ignore)]
    fn u8_proptest(i in u8::min_value()..u8::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn i8_proptest(i in i8::min_value()..i8::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn u16_proptest(i in u16::min_value()..u16::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn i16_proptest(i in i16::min_value()..i16::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn u32_proptest(i in u32::min_value()..u32::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn i32_proptest(i in i32::min_value()..i32::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn u64_proptest(i in u64::min_value()..u64::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn i64_proptest(i in i64::min_value()..i64::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn u128_proptest(i in u128::min_value()..u128::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn i128_proptest(i in i128::min_value()..i128::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn usize_proptest(i in usize::min_value()..usize::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    fn isize_proptest(i in isize::min_value()..isize::max_value()) {
        prop_assert_eq!(i, roundtrip(i));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn u8_proptest_radix(i in u8::min_value()..u8::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn i8_proptest_radix(i in i8::min_value()..i8::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn u16_proptest_radix(i in u16::min_value()..u16::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn i16_proptest_radix(i in i16::min_value()..i16::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn u32_proptest_radix(i in u32::min_value()..u32::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn i32_proptest_radix(i in i32::min_value()..i32::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn u64_proptest_radix(i in u64::min_value()..u64::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn i64_proptest_radix(i in i64::min_value()..i64::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn u128_proptest_radix(i in u128::min_value()..u128::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn i128_proptest_radix(i in i128::min_value()..i128::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn usize_proptest_radix(i in usize::min_value()..usize::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }

    #[test]
    #[cfg_attr(miri, ignore)]
    #[cfg(feature = "radix")]
    fn isize_proptest_radix(i in isize::min_value()..isize::max_value(), radix in 2u32..=36) {
        prop_assert_eq!(i, roundtrip_radix(i, radix));
    }
}

#[test]
#[should_panic]
fn i8_buffer_test() {
    let mut buffer = [b'\x00'; i8::FORMATTED_SIZE_DECIMAL - 1];
    12i8.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn i16_buffer_test() {
    let mut buffer = [b'\x00'; i16::FORMATTED_SIZE_DECIMAL - 1];
    12i16.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn i32_buffer_test() {
    let mut buffer = [b'\x00'; i32::FORMATTED_SIZE_DECIMAL - 1];
    12i32.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn i64_buffer_test() {
    let mut buffer = [b'\x00'; i64::FORMATTED_SIZE_DECIMAL - 1];
    12i64.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn i128_buffer_test() {
    let mut buffer = [b'\x00'; i128::FORMATTED_SIZE_DECIMAL - 1];
    12i128.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn isize_buffer_test() {
    let mut buffer = [b'\x00'; isize::FORMATTED_SIZE_DECIMAL - 1];
    12isize.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn u8_buffer_test() {
    let mut buffer = [b'\x00'; u8::FORMATTED_SIZE_DECIMAL - 1];
    12i8.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn u16_buffer_test() {
    let mut buffer = [b'\x00'; u16::FORMATTED_SIZE_DECIMAL - 1];
    12i16.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn u32_buffer_test() {
    let mut buffer = [b'\x00'; u32::FORMATTED_SIZE_DECIMAL - 1];
    12i32.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn u64_buffer_test() {
    let mut buffer = [b'\x00'; u64::FORMATTED_SIZE_DECIMAL - 1];
    12i64.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn u128_buffer_test() {
    let mut buffer = [b'\x00'; u128::FORMATTED_SIZE_DECIMAL - 1];
    12i128.to_lexical(&mut buffer);
}

#[test]
#[should_panic]
fn usize_buffer_test() {
    let mut buffer = [b'\x00'; usize::FORMATTED_SIZE_DECIMAL - 1];
    12usize.to_lexical(&mut buffer);
}
