use std::fmt::{Display, Formatter};
use std::ops::Range;

use crate::Kind;

/// Represents a lexical token.
#[derive(Copy, Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
pub struct Token {
    kind: Kind,
    position: usize,
    length: usize,
}

impl Token {
    //! Constructors

    /// Creates a new token.
    pub const fn new(kind: Kind, position: usize, length: usize) -> Self {
        Self {
            kind,
            position,
            length,
        }
    }
}

impl Token {
    //! Properties

    /// Gets the kind of token.
    pub const fn kind(&self) -> Kind {
        self.kind
    }

    /// Gets the 0-indexed position of the first byte within the source code.
    pub const fn position(&self) -> usize {
        self.position
    }

    /// Gets the length. (in bytes)
    pub const fn length(&self) -> usize {
        self.length
    }

    /// Gets the limit. (position + length)
    pub const fn limit(&self) -> usize {
        self.position + self.length
    }

    /// Gets the range.
    pub const fn range(&self) -> Range<usize> {
        self.position..self.limit()
    }

    /// Gets the value from the source code.
    ///
    /// # Panics
    /// Panics if the token is invalid in the source code.
    pub fn value<'a>(&self, source: &'a str) -> &'a str {
        &source[self.range()]
    }
}

impl Display for Token {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(
            f,
            "kind={} pos={} len={}",
            self.kind, self.position, self.length
        )
    }
}
