use std::fmt::{Display, Formatter};

use crate::{Kind, Token};

/// Represents tokenized source code.
#[derive(Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
pub struct Lex<'a> {
    source: &'a str,
    tokens: Vec<Token>,
}

impl<'a> Lex<'a> {
    //! Constructors

    /// Creates a new lex.
    pub const fn new(source: &'a str, tokens: Vec<Token>) -> Self {
        Self { source, tokens }
    }
}

impl<'a> Lex<'a> {
    //! Properties

    /// Gets the raw source code.
    pub const fn source(&self) -> &str {
        self.source
    }

    /// Gets the tokens.
    pub fn tokens(&self) -> &[Token] {
        &self.tokens
    }
}

impl<'a> Lex<'a> {
    //! Exporting

    /// Exports the tokens.
    pub fn export_tokens(self) -> Vec<Token> {
        self.tokens
    }
}

impl<'a> Lex<'a> {
    //! Display

    // Formats the value with standard US-ASCII characters.
    fn fmt_value(&self, f: &mut Formatter<'_>, value: &str) -> std::fmt::Result {
        for c in value.as_bytes() {
            match *c {
                b' ' => write!(f, "s")?,
                b'\t' => write!(f, "t")?,
                b'\r' => write!(f, "r")?,
                b'\n' => write!(f, "n")?,
                c if c.is_ascii() && !c.is_ascii_control() => write!(f, "{}", c as char)?,
                _ => write!(f, "?")?,
            }
        }
        Ok(())
    }
}

impl<'a> Display for Lex<'a> {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "line 0:\n")?;
        let mut line: usize = 0;
        let mut line_position: usize = 0;
        for token in &self.tokens {
            write!(f, "\t{: >5}  ", line_position)?;
            write!(f, "{: <12}:  ", token.kind().display_str())?;
            self.fmt_value(f, token.value(self.source))?;
            write!(f, "\n")?;
            if token.kind() == Kind::LineEnding {
                line_position = 0;
                line += 1;
                write!(f, "line {}:\n", line)?;
            } else {
                line_position += token.length();
            }
        }
        Ok(())
    }
}
