use std::fmt::{Display, Formatter};

/// Represents a kind of lexical token.
#[derive(Copy, Clone, Ord, PartialOrd, Eq, PartialEq, Hash, Debug)]
pub enum Kind {
    /// A sequence of non-US-ASCII characters. (code points above 127)
    NonAscii,

    /// A single US-ASCII line-ending. (CR, LF, or CLRF)
    LineEnding,

    // A sequence of US-ASCII whitespace characters. (spaces and tabs)
    Whitespace,

    /// A sequence of US-ASCII control characters. (excluding CR, LF, and TAB)
    Controls,

    /// A sequence of US-ASCII letters, numbers, and underscores.
    Symbol,

    /// A single special US-ASCII character not mentioned above.
    Special(u8),
}

impl Kind {
    //! Display

    /// Gets the display string.
    pub const fn display_str(&self) -> &'static str {
        match self {
            Kind::NonAscii => "non-ascii",
            Kind::LineEnding => "line-ending",
            Kind::Whitespace => "whitespace",
            Kind::Controls => "controls",
            Kind::Symbol => "symbol",
            Kind::Special(_) => "special",
        }
    }
}

impl Display for Kind {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        write!(f, "{}", self.display_str())
    }
}
