use super::*;

use serde::ser::{Serialize, SerializeMap, Serializer};

#[derive(Default, Serialize)]
pub struct Query<'a> {
    #[serde(skip_serializing_if = "Option::is_none")]
    and: Option<Vec<QueryClause<'a>>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    not: Option<Vec<QueryClause<'a>>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    or: Option<Vec<QueryClause<'a>>>,
}

impl<'a> Query<'a> {
    pub fn new() -> Query<'a> {
        Query::default()
    }

    pub fn and<F: FnOnce(&mut ClauseBuilder)>(mut self, op: F) -> Query<'a> {
        let mut builder = ClauseBuilder::new();
        op(&mut builder);
        self.and
            .get_or_insert_default()
            .append(&mut builder.build());
        self
    }

    pub fn not<F: FnOnce(&mut ClauseBuilder)>(mut self, op: F) -> Query<'a> {
        let mut builder = ClauseBuilder::new();
        op(&mut builder);
        self.not
            .get_or_insert_default()
            .append(&mut builder.build());
        self
    }

    pub fn or<F: FnOnce(&mut ClauseBuilder)>(mut self, op: F) -> Query<'a> {
        let mut builder = ClauseBuilder::new();
        op(&mut builder);
        self.or.get_or_insert_default().append(&mut builder.build());
        self
    }
}

pub struct ClauseBuilder<'a> {
    clauses: Vec<QueryClause<'a>>,
}

impl<'a> ClauseBuilder<'a> {
    pub(self) fn new() -> ClauseBuilder<'a> {
        ClauseBuilder {
            clauses: Vec::new(),
        }
    }

    pub(self) fn build(self) -> Vec<QueryClause<'a>> {
        self.clauses
    }

    pub fn done(&mut self) {}

    pub fn query(&mut self, query: Query<'a>) -> &mut ClauseBuilder<'a> {
        self.clauses.push(QueryClause::Query(query));
        self
    }

    pub fn eq<T>(&mut self, key: &'a str, value: T) -> &mut ClauseBuilder<'a>
    where
        T: Serialize,
    {
        self.clauses
            .push(QueryClause::Clause(clause::ClauseCase::Match(
                key,
                serde_json::json!(value),
            )));
        self
    }

    pub fn lt<T>(&mut self, key: &'a str, value: T) -> &mut ClauseBuilder<'a>
    where
        T: Serialize,
    {
        self.clauses
            .push(QueryClause::Clause(clause::ClauseCase::Range(
                key,
                clause::RangeCase::Lt(serde_json::json!(value)),
            )));
        self
    }

    pub fn lte<T>(&mut self, key: &'a str, value: T) -> &mut ClauseBuilder<'a>
    where
        T: Serialize,
    {
        self.clauses
            .push(QueryClause::Clause(clause::ClauseCase::Range(
                key,
                clause::RangeCase::Lte(serde_json::json!(value)),
            )));
        self
    }

    pub fn gt<T>(&mut self, key: &'a str, value: T) -> &mut ClauseBuilder<'a>
    where
        T: Serialize,
    {
        self.clauses
            .push(QueryClause::Clause(clause::ClauseCase::Range(
                key,
                clause::RangeCase::Gt(serde_json::json!(value)),
            )));
        self
    }

    pub fn gte<T>(&mut self, key: &'a str, value: T) -> &mut ClauseBuilder<'a>
    where
        T: Serialize,
    {
        self.clauses
            .push(QueryClause::Clause(clause::ClauseCase::Range(
                key,
                clause::RangeCase::Gte(serde_json::json!(value)),
            )));
        self
    }
}

enum QueryClause<'a> {
    Clause(clause::ClauseCase<'a>),
    Query(Query<'a>),
}

impl<'a> Serialize for QueryClause<'a> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        match self {
            QueryClause::Clause(v) => serializer.serialize_newtype_struct("Clause", v),
            QueryClause::Query(v) => {
                let mut map = serializer.serialize_map(Some(1))?;
                map.serialize_entry("query", v)?;
                map.end()
            }
        }
    }
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn query() {
        let q = Query {
            and: Some(vec![
                QueryClause::Clause(clause::ClauseCase::Range(
                    "t-id",
                    clause::RangeCase::Gt(serde_json::json!(1)),
                )),
                QueryClause::Query(Query {
                    and: Some(vec![QueryClause::Clause(clause::ClauseCase::Range(
                        "t-id",
                        clause::RangeCase::Lt(serde_json::json!(1)),
                    ))]),
                    not: None,
                    or: Some(vec![QueryClause::Clause(clause::ClauseCase::Match(
                        "t-id",
                        serde_json::json!("qians"),
                    ))]),
                }),
            ]),
            not: None,
            or: None,
        };

        let q = serde_json::to_value(&q).unwrap().to_string();
        let qv = serde_json::json!({
                "and": [
                    {"range": {"t-id": {"gt": 1}}},
                    {"query": {
                        "and": [
                            {"range": {"t-id": {"lt": 1}}}
                        ],
                        "or": [
                            {"match": {"t-id": "qians"}}
                        ]
                    }}
                ]
            }
        )
        .to_string();
        assert_eq!(q, qv);
    }
}
