use super::*;

#[derive(Default, Serialize)]
pub struct Conditions<'a> {
    #[serde(skip_serializing_if = "Option::is_none")]
    pub(super) pagination: Option<Pagination>,
    #[serde(rename = "order", skip_serializing_if = "Option::is_none")]
    pub(super) orders: Option<HashMap<&'a str, OrderCase>>,

    pub(super) fields: Vec<&'a str>,
    pub(super) query: query::Query<'a>,
    #[serde(flatten, skip_serializing_if = "Option::is_none")]
    pub(super) aggs: Option<aggregation::Aggs<'a>>,
}

impl<'a> Conditions<'a> {
    pub(super) fn from_query(query: query::Query<'a>) -> Conditions<'a> {
        Conditions {
            query,
            ..Default::default()
        }
    }

    pub(super) fn aggs(&mut self, aggs: aggregation::Aggs<'a>) {
        self.aggs = Some(aggs)
    }

    pub(super) fn pagination(&mut self, page: u32, page_size: u32) {
        self.pagination = Some(Pagination { page, page_size })
    }

    pub(super) fn desc(&mut self, order: &'a str) {
        if let Some(orders) = &mut self.orders {
            orders.insert(order, OrderCase::Desc);
        } else {
            self.orders = Some(HashMap::from([(order, OrderCase::Desc)]));
        }
    }

    pub(super) fn asc(&mut self, order: &'a str) {
        if let Some(orders) = &mut self.orders {
            orders.insert(order, OrderCase::Asc);
        } else {
            self.orders = Some(HashMap::from([(order, OrderCase::Asc)]));
        }
    }

    pub(super) fn field(&mut self, field: &'a str) {
        self.fields.push(field);
    }

    pub(super) fn fields(&mut self, fields: Vec<&'a str>) {
        fields.iter().for_each(|&field| {
            self.fields.push(field);
        });
    }
}

#[derive(Serialize)]
pub struct Pagination {
    pub(crate) page: u32,
    #[serde(rename = "pageSize")]
    pub(crate) page_size: u32,
}

#[derive(Serialize)]
pub(crate) enum OrderCase {
    #[serde(rename = "desc")]
    Desc,
    #[serde(rename = "asc")]
    Asc,
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn order() {
        let r = HashMap::from([("t-id", OrderCase::Desc)]);
        let r = serde_json::to_value(&r).unwrap().to_string();
        let rv = serde_json::json!({"t-id": "desc"}).to_string();
        assert_eq!(r, rv);

        let r = HashMap::from([("t-id", OrderCase::Asc)]);
        let r = serde_json::to_value(&r).unwrap().to_string();
        let rv = serde_json::json!({"t-id": "asc"}).to_string();
        assert_eq!(r, rv);
    }
}
