use super::*;

use serde::ser::{Serialize, SerializeMap, Serializer};
pub(super) enum ClauseCase<'a> {
    Match(&'a str, serde_json::Value),
    Range(&'a str, RangeCase),
}

impl<'a> Serialize for ClauseCase<'a> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        let mut map = serializer.serialize_map(Some(1))?;
        match self {
            ClauseCase::Match(k, v) => {
                let vv = HashMap::from([(k, v)]);
                map.serialize_entry("match", &vv)?
            }
            ClauseCase::Range(k, v) => {
                let vv = HashMap::from([(k, v)]);
                map.serialize_entry("range", &vv)?
            }
        };
        map.end()
    }
}

#[derive(Serialize)]
pub(super) enum RangeCase {
    // GT: >
    #[serde(rename = "gt")]
    Gt(serde_json::Value),
    // GTE: >=
    #[serde(rename = "gte")]
    Gte(serde_json::Value),
    // LT: <
    #[serde(rename = "lt")]
    Lt(serde_json::Value),
    // LTE: <=
    #[serde(rename = "lte")]
    Lte(serde_json::Value),
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn range_case() {
        let gt = RangeCase::Gt(serde_json::json!(1));
        let gt = serde_json::to_value(&gt).unwrap().to_string();
        let gtv = serde_json::json!({"gt": 1}).to_string();
        assert_eq!(gt, gtv);

        let gte = RangeCase::Gte(serde_json::json!("1"));
        let gte = serde_json::to_value(&gte).unwrap().to_string();
        let gtev = serde_json::json!({"gte": "1"}).to_string();
        assert_eq!(gte, gtev);

        let lt = RangeCase::Lt(serde_json::json!(1.1));
        let lt = serde_json::to_value(&lt).unwrap().to_string();
        let ltv = serde_json::json!({"lt": 1.1}).to_string();
        assert_eq!(lt, ltv);

        let lte = RangeCase::Lte(serde_json::json!("abc"));
        let lte = serde_json::to_value(&lte).unwrap().to_string();
        let ltev = serde_json::json!({"lte": "abc"}).to_string();
        assert_eq!(lte, ltev);
    }

    #[test]
    fn clause_case() {
        let r = ClauseCase::Range("t-id", RangeCase::Gt(serde_json::json!(1)));
        let r = serde_json::to_value(&r).unwrap().to_string();
        let rv = serde_json::json!({"range": {"t-id": {"gt": 1}}}).to_string();
        assert_eq!(r, rv);

        let m = ClauseCase::Match("t-id", serde_json::json!(1));
        let m = serde_json::to_value(&m).unwrap().to_string();
        let mv = serde_json::json!({"match": {"t-id": 1}}).to_string();
        assert_eq!(m, mv);
    }
}
