use super::*;

pub struct Project<'a> {
    pub(super) project: &'a str,
}

impl<'a> Project<'a> {
    pub fn table(self, table: &'a str) -> Table<'a> {
        Table {
            project: self.project,
            table,
        }
    }
}

pub struct Table<'a> {
    project: &'a str,
    table: &'a str,
}

impl<'a> Table<'a> {
    pub fn create(self) -> CreateBuilder<'a> {
        CreateBuilder {
            project: self.project,
            table: self.table,
            data: Vec::new(),
        }
    }

    pub fn get(self, id: &'a str) -> GetBuilder<'a> {
        GetBuilder {
            project: self.project,
            table: self.table,
            id,
            fields: Vec::new(),
        }
    }

    pub fn set(self, id: &'a str) -> SetBuilder<'a> {
        SetBuilder {
            project: self.project,
            table: self.table,
            id,
        }
    }

    pub fn query(self, query: query::Query<'a>) -> QueryBuilder<'a> {
        QueryBuilder {
            project: self.project,
            table: self.table,
            conditions: conditions::Conditions::from_query(query),
        }
    }

    pub fn update_by_query(self, query: query::Query<'a>) -> UpdateByQueryBuilder<'a> {
        UpdateByQueryBuilder {
            project: self.project,
            table: self.table,
            query,
        }
    }
}

pub struct CreateBuilder<'a> {
    project: &'a str,
    table: &'a str,

    data: Vec<serde_json::Value>,
}

impl<'a> CreateBuilder<'a> {
    pub fn create_one(mut self, v: serde_json::Value) -> CreateBuilder<'a> {
        self.data.push(v);
        self
    }

    pub fn done(self) -> Request<'a> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::Create { data: self.data },
        }
    }
}

pub struct GetBuilder<'a> {
    project: &'a str,
    table: &'a str,

    id: &'a str,
    fields: Vec<&'a str>,
}

impl<'a> GetBuilder<'a> {
    pub fn field(mut self, field: &'a str) -> GetBuilder<'a> {
        self.fields.push(field);
        self
    }

    pub fn fields(mut self, fields: Vec<&'a str>) -> GetBuilder<'a> {
        fields.iter().for_each(|&field| {
            self.fields.push(field);
        });
        self
    }

    pub fn done(self) -> Request<'a> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::Get {
                id: self.id,
                fields: self.fields,
            },
        }
    }
}

pub struct SetBuilder<'a> {
    project: &'a str,
    table: &'a str,

    id: &'a str,
}

impl<'a> SetBuilder<'a> {
    pub fn done(self, data: serde_json::Value) -> Request<'a> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::Set { id: self.id, data },
        }
    }
}

pub struct QueryBuilder<'a> {
    project: &'a str,
    table: &'a str,

    conditions: conditions::Conditions<'a>,
}

impl<'a> QueryBuilder<'a> {
    pub fn aggs(mut self, aggs: aggregation::Aggs<'a>) -> QueryBuilder<'a> {
        self.conditions.aggs(aggs);
        self
    }

    pub fn pagination(mut self, page: u32, page_size: u32) -> QueryBuilder<'a> {
        self.conditions.pagination(page, page_size);
        self
    }

    pub fn desc(mut self, order: &'a str) -> QueryBuilder<'a> {
        self.conditions.desc(order);
        self
    }

    pub fn asc(mut self, order: &'a str) -> QueryBuilder<'a> {
        self.conditions.asc(order);
        self
    }

    pub fn field(mut self, field: &'a str) -> QueryBuilder<'a> {
        self.conditions.field(field);
        self
    }
    pub fn fields(mut self, fields: Vec<&'a str>) -> QueryBuilder<'a> {
        self.conditions.fields(fields);
        self
    }
    pub fn done(self) -> Request<'a> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::Query {
                conditions: self.conditions,
            },
        }
    }
}

pub struct UpdateByQueryBuilder<'a> {
    project: &'a str,
    table: &'a str,

    query: query::Query<'a>,
}

impl<'a> UpdateByQueryBuilder<'a> {
    pub fn done(self, data: serde_json::Value) -> Request<'a> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::UpdateByQuery {
                query: self.query,
                data,
            },
        }
    }
}
