use super::*;
use serde::ser::{Serialize, SerializeMap, Serializer};

#[derive(Default, Serialize)]
pub struct Aggs<'a> {
    aggs: HashMap<&'a str, AggrCase<'a>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    having: Option<Vec<AggrCase<'a>>>,
}

impl<'a> Aggs<'a> {
    pub fn new() -> Aggs<'a> {
        Aggs::default()
    }

    pub fn aggr<F: FnOnce(&mut AggrBuilder)>(mut self, name: &'a str, op: F) -> Aggs<'a> {
        let mut builder = AggrBuilder::new();
        op(&mut builder);
        self.aggs.insert(name, builder.build());
        self
    }

    pub fn having() {
        todo!()
    }
}

pub struct AggrBuilder<'a> {
    aggr: AggrCase<'a>,
}

impl<'a> AggrBuilder<'a> {
    pub(self) fn new() -> AggrBuilder<'a> {
        AggrBuilder {
            aggr: AggrCase::default(),
        }
    }

    pub(self) fn build(self) -> AggrCase<'a> {
        self.aggr
    }

    pub fn done(&mut self) {}

    pub fn terms(&mut self, field: &'a str) -> &mut AggrBuilder<'a> {
        self.aggr.metrics = Some(Metrics::Terms(field));
        self
    }

    pub fn count(&mut self, field: &'a str) -> &mut AggrBuilder<'a> {
        self.aggr.metrics = Some(Metrics::Count(field));
        self
    }

    pub fn avg(&mut self, field: &'a str) -> &mut AggrBuilder<'a> {
        self.aggr.metrics = Some(Metrics::Avg(field));
        self
    }

    pub fn min(&mut self, field: &'a str) -> &mut AggrBuilder<'a> {
        self.aggr.metrics = Some(Metrics::Min(field));
        self
    }

    pub fn max(&mut self, field: &'a str) -> &mut AggrBuilder<'a> {
        self.aggr.metrics = Some(Metrics::Max(field));
        self
    }
    pub fn sum(&mut self, field: &'a str) -> &mut AggrBuilder<'a> {
        self.aggr.metrics = Some(Metrics::Sum(field));
        self
    }

    pub fn aggs(&mut self, aggs: Aggs<'a>) -> &mut AggrBuilder<'a> {
        self.aggr.aggs = Some(aggs);
        self
    }
}

#[derive(Default, Serialize)]
struct AggrCase<'a> {
    #[serde(flatten)]
    metrics: Option<Metrics<'a>>,
    #[serde(flatten)]
    aggs: Option<Aggs<'a>>,
}

enum Metrics<'a> {
    Terms(&'a str),
    Count(&'a str),
    Avg(&'a str),
    Min(&'a str),
    Max(&'a str),
    Sum(&'a str),
}

impl<'a> Serialize for Metrics<'a> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        let mut map = serializer.serialize_map(Some(1))?;
        match self {
            Metrics::Terms(f) => {
                let vv = HashMap::from([("field", f)]);
                map.serialize_entry("terms", &vv)?
            }
            Metrics::Count(f) => {
                let vv = HashMap::from([("field", f)]);
                map.serialize_entry("count", &vv)?
            }
            Metrics::Avg(f) => {
                let vv = HashMap::from([("field", f)]);
                map.serialize_entry("avg", &vv)?
            }
            Metrics::Min(f) => {
                let vv = HashMap::from([("field", f)]);
                map.serialize_entry("min", &vv)?
            }
            Metrics::Max(f) => {
                let vv = HashMap::from([("field", f)]);
                map.serialize_entry("max", &vv)?
            }
            Metrics::Sum(f) => {
                let vv = HashMap::from([("field", f)]);
                map.serialize_entry("sum", &vv)?
            }
        };
        map.end()
    }
}

#[cfg(test)]
mod test {
    use super::*;
    use std::collections::HashMap;
    #[test]
    fn aggs() {
        let aggsv = serde_json::json!({
            "aggs": {
                "aggr1": {
                    "terms": {"field": "name"},
                    "aggs": {"aggr2": {"avg": {"field": "age"}}}
                }
            }
        })
        .to_string();
        let aggs = Aggs {
            aggs: HashMap::from([(
                "aggr1",
                AggrCase {
                    metrics: Some(Metrics::Terms("name")),
                    aggs: Some(Aggs {
                        aggs: HashMap::from([(
                            "aggr2",
                            AggrCase {
                                metrics: Some(Metrics::Avg("age")),
                                aggs: None,
                            },
                        )]),
                        having: None,
                    }),
                },
            )]),
            having: None,
        };
        let aggs = serde_json::to_value(&aggs).unwrap().to_string();
        assert_eq!(aggs, aggsv);
    }

    #[test]
    fn aggr() {
        let terms = Metrics::Terms("aggr");
        let terms = serde_json::to_value(&terms).unwrap().to_string();
        let termsv = serde_json::json!({"terms": {"field": "aggr"}}).to_string();
        assert_eq!(terms, termsv);

        let count = Metrics::Count("aggr");
        let count = serde_json::to_value(&count).unwrap().to_string();
        let countv = serde_json::json!({"count": {"field": "aggr"}}).to_string();
        assert_eq!(count, countv);

        let avg = Metrics::Avg("aggr");
        let avg = serde_json::to_value(&avg).unwrap().to_string();
        let avgv = serde_json::json!({"avg": {"field": "aggr"}}).to_string();
        assert_eq!(avg, avgv);

        let min = Metrics::Min("aggr");
        let min = serde_json::to_value(&min).unwrap().to_string();
        let minv = serde_json::json!({"min": {"field": "aggr"}}).to_string();
        assert_eq!(min, minv);

        let max = Metrics::Max("aggr");
        let max = serde_json::to_value(&max).unwrap().to_string();
        let maxv = serde_json::json!({"max": {"field": "aggr"}}).to_string();
        assert_eq!(max, maxv);

        let sum = Metrics::Sum("aggr");
        let sum = serde_json::to_value(&sum).unwrap().to_string();
        let sumv = serde_json::json!({"sum": {"field": "aggr"}}).to_string();
        assert_eq!(sum, sumv);
    }
}
