#![feature(option_get_or_insert_default)]

mod error;
mod request;
mod response;

pub use request::Aggs;
pub use request::Query;
pub use request::Request;
pub use response::Response;

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn query() {
        let req = Request::project("project")
            .table("table")
            .query(Query::new().and(|b| {
                b.gt("t-id", 1).query(
                    Query::new()
                        .and(|b| b.lt("t-id", 1).done())
                        .or(|b| b.eq("t-id", "qians").done()),
                );
            }))
            .desc("t-id")
            .fields(vec!["t-id", "t-name"])
            .aggs(Aggs::new().aggr("space", |b| {
                b.terms("path_code")
                    .aggs(Aggs::new().aggr("receiver", |b| b.terms("receiver").done()))
                    .done()
            }))
            .pagination(1, 4)
            .done()
            .to_string();

        let reqj = serde_json::json!({
            "auth": 2,
            "project": "project",
            "table": "table",
            "conditions": {
                "pagination": {
                    "page": 1,
                    "pageSize": 4
                },
                "order": {
                    "t-id": "desc"
                },
                "fields": ["t-id", "t-name"],
                "query": {
                    "and": [
                        {"range": {"t-id": {"gt": 1}}},
                        {"query": {
                            "and": [
                                {"range": {"t-id": {"lt": 1}}}
                            ],
                            "or": [
                                {"match": {"t-id": "qians"}}
                            ]
                        }}
                    ]
                },
                "aggs": {
                    "space": {
                        "terms": {
                            "field": "path_code"
                        },
                        "aggs": {
                            "receiver": {
                                "terms": {
                                    "field": "receiver"
                                }
                            }
                        }
                    }
                }
            }
        })
        .to_string();
        assert_eq!(req, reqj);
    }

    #[test]
    fn update_by_query() {
        let req = Request::project("project")
            .table("table")
            .update_by_query(Query::new().and(|builder| builder.eq("id", "2").done()))
            .done(serde_json::json!({
                "t-name": "hhh",
                "t-age": 10
            }))
            .to_string();

        let qv = serde_json::json!({
            "auth": 2,
            "project": "project",
            "table": "table",
            "query": {
                "and": [
                    {"match": {"id": "2"}}
                ]
            },
            "data": {
                "t-name": "hhh",
                "t-age": 10
            }
        })
        .to_string();
        assert_eq!(req, qv);
    }

    #[test]
    fn get() {
        let req = Request::project("project")
            .table("table")
            .get("2")
            .fields(vec!["t-name", "t-age"])
            .done()
            .to_string();

        let reqj = serde_json::json!({
            "auth": 2,
            "project": "project",
            "table": "table",
            "fields": ["t-name", "t-age"],
            "id": "2"
        })
        .to_string();
        assert_eq!(req, reqj)
    }

    #[test]
    fn set() {
        let req = Request::project("project")
            .table("table")
            .set("2")
            .done(serde_json::json!({
                "t-name": "h",
                "t-age": 10
            }))
            .to_string();

        let reqj = serde_json::json!({
            "auth": 2,
            "project": "project",
            "table": "table",
            "id": "2",
            "data": {
                "t-name": "h",
                "t-age": 10
            }
        })
        .to_string();
        assert_eq!(req, reqj)
    }

    #[test]
    fn create() {
        let req = Request::project("project")
            .table("table")
            .create()
            .create_one(serde_json::json!({
                "t-name": "h",
                "t-age": 10
            }))
            .create_one(serde_json::json!({
                "t-name": "hhh",
                "t-age": 10
            }))
            .done()
            .to_string();

        let reqj = serde_json::json!({
            "auth": 2,
            "project": "project",
            "table": "table",
            "data":[
                {
                    "t-name": "h",
                    "t-age": 10
                },
                {
                    "t-name": "hhh",
                    "t-age": 10
                }
            ]
        })
        .to_string();
        assert_eq!(req, reqj)
    }
}
