use super::*;

#[derive(Default, Serialize)]
pub struct Conditions<'a> {
    #[serde(skip_serializing_if = "Option::is_none")]
    pub(crate) pagination: Option<Pagination>,
    #[serde(rename = "order", skip_serializing_if = "Option::is_none")]
    pub(crate) orders: Option<HashMap<&'a str, OrderCase>>,

    pub(crate) fields: Vec<&'a str>,
    pub(crate) query: query::Query<'a>,
}

impl<'a> Conditions<'a> {
    pub fn from_query(query: query::Query<'a>) -> Conditions<'a> {
        Conditions {
            query,
            ..Default::default()
        }
    }

    pub fn set_pagination(&mut self, page: u32, page_size: u32) {
        self.pagination = Some(Pagination { page, page_size })
    }

    pub fn desc(&mut self, order: &'a str) {
        if let Some(orders) = &mut self.orders {
            orders.insert(order, OrderCase::Desc);
        } else {
            self.orders = Some(HashMap::from([(order, OrderCase::Desc)]));
        }
    }

    pub fn asc(&mut self, order: &'a str) {
        if let Some(orders) = &mut self.orders {
            orders.insert(order, OrderCase::Asc);
        } else {
            self.orders = Some(HashMap::from([(order, OrderCase::Asc)]));
        }
    }

    pub fn add_field(&mut self, field: &'a str) {
        self.fields.push(field);
    }
    pub fn add_fields(&mut self, fields: &'a [String]) {
        fields.iter().for_each(|field| {
            self.fields.push(field);
        });
    }
}

#[derive(Serialize)]
pub struct Pagination {
    pub(crate) page: u32,
    #[serde(rename = "pageSize")]
    pub(crate) page_size: u32,
}

#[derive(Serialize)]
pub(crate) enum OrderCase {
    #[serde(rename = "desc")]
    Desc,
    #[serde(rename = "asc")]
    Asc,
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn order() {
        let r = HashMap::from([("t-id", OrderCase::Desc)]);
        let r = serde_json::to_value(&r).unwrap().to_string();
        let rv = serde_json::json!({"t-id": "desc"}).to_string();
        assert_eq!(r, rv);

        let r = HashMap::from([("t-id", OrderCase::Asc)]);
        let r = serde_json::to_value(&r).unwrap().to_string();
        let rv = serde_json::json!({"t-id": "asc"}).to_string();
        assert_eq!(r, rv);
    }
}
