use super::*;

use serde::ser::{Serialize, SerializeMap, Serializer};

#[derive(Default, Serialize)]
pub struct Query<'a> {
    #[serde(skip_serializing_if = "Option::is_none")]
    and: Option<Vec<Clause<'a>>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    not: Option<Vec<Clause<'a>>>,
    #[serde(skip_serializing_if = "Option::is_none")]
    or: Option<Vec<Clause<'a>>>,
}

impl<'a> Query<'a> {
    pub fn new() -> Query<'a> {
        Query::default()
    }

    pub fn and(self) -> ClauseBuilder<'a> {
        ClauseBuilder {
            q_yield: self,
            query_case: QueryCase::And,
        }
    }

    pub fn not(self) -> ClauseBuilder<'a> {
        ClauseBuilder {
            q_yield: self,
            query_case: QueryCase::Not,
        }
    }

    pub fn or(self) -> ClauseBuilder<'a> {
        ClauseBuilder {
            q_yield: self,
            query_case: QueryCase::Or,
        }
    }
}

enum QueryCase {
    And,
    Not,
    Or,
}

impl QueryCase {
    pub(self) fn process<'a>(self, mut query: Query<'a>, clause: Clause<'a>) -> Query<'a> {
        match self {
            QueryCase::And => query.and.get_or_insert_default(),
            QueryCase::Not => query.not.get_or_insert_default(),
            QueryCase::Or => query.or.get_or_insert_default(),
        }
        .push(clause);
        query
    }
}

pub struct ClauseBuilder<'a> {
    query_case: QueryCase,

    q_yield: Query<'a>,
}

impl<'a> ClauseBuilder<'a> {
    pub fn query(self, query: Query<'a>) -> Query<'a> {
        self.query_case.process(self.q_yield, Clause::Query(query))
    }

    pub fn eq<T>(self, key: &'a str, value: T) -> Query<'a>
    where
        T: Serialize,
    {
        self.query_case.process(
            self.q_yield,
            Clause::Clause(ClauseCase::Match(key, serde_json::json!(value))),
        )
    }

    pub fn lt<T>(self, key: &'a str, value: T) -> Query<'a>
    where
        T: Serialize,
    {
        self.query_case.process(
            self.q_yield,
            Clause::Clause(ClauseCase::Range(
                key,
                RangeCase::Lt(serde_json::json!(value)),
            )),
        )
    }

    pub fn lte<T>(self, key: &'a str, value: T) -> Query<'a>
    where
        T: Serialize,
    {
        self.query_case.process(
            self.q_yield,
            Clause::Clause(ClauseCase::Range(
                key,
                RangeCase::Lte(serde_json::json!(value)),
            )),
        )
    }

    pub fn gt<T>(self, key: &'a str, value: T) -> Query<'a>
    where
        T: Serialize,
    {
        self.query_case.process(
            self.q_yield,
            Clause::Clause(ClauseCase::Range(
                key,
                RangeCase::Gt(serde_json::json!(value)),
            )),
        )
    }

    pub fn gte<T>(self, key: &'a str, value: T) -> Query<'a>
    where
        T: Serialize,
    {
        self.query_case.process(
            self.q_yield,
            Clause::Clause(ClauseCase::Range(
                key,
                RangeCase::Gte(serde_json::json!(value)),
            )),
        )
    }
}

enum Clause<'a> {
    Clause(ClauseCase<'a>),
    Query(Query<'a>),
}

impl<'a> Serialize for Clause<'a> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        match self {
            Clause::Clause(v) => serializer.serialize_newtype_struct("Clause", v),
            Clause::Query(v) => {
                let mut map = serializer.serialize_map(Some(1))?;
                map.serialize_entry("query", v)?;
                map.end()
            }
        }
    }
}

enum ClauseCase<'a> {
    Match(&'a str, serde_json::Value),
    Range(&'a str, RangeCase),
}

impl<'a> Serialize for ClauseCase<'a> {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: Serializer,
    {
        let mut map = serializer.serialize_map(Some(1))?;
        match self {
            ClauseCase::Match(k, v) => {
                let vv = HashMap::from([(k, v)]);
                map.serialize_entry("match", &vv)?
            }
            ClauseCase::Range(k, v) => {
                let vv = HashMap::from([(k, v)]);
                map.serialize_entry("range", &vv)?
            }
        };
        map.end()
    }
}

#[derive(Serialize)]
enum RangeCase {
    // GT: >
    #[serde(rename = "gt")]
    Gt(serde_json::Value),
    // GTE: >=
    #[serde(rename = "gte")]
    Gte(serde_json::Value),
    // LT: <
    #[serde(rename = "lt")]
    Lt(serde_json::Value),
    // LTE: <=
    #[serde(rename = "lte")]
    Lte(serde_json::Value),
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn range_case() {
        let gt = RangeCase::Gt(serde_json::json!(1));
        let gt = serde_json::to_value(&gt).unwrap().to_string();
        let gtv = serde_json::json!({"gt": 1}).to_string();
        assert_eq!(gt, gtv);

        let gte = RangeCase::Gte(serde_json::json!("1"));
        let gte = serde_json::to_value(&gte).unwrap().to_string();
        let gtev = serde_json::json!({"gte": "1"}).to_string();
        assert_eq!(gte, gtev);

        let lt = RangeCase::Lt(serde_json::json!(1.1));
        let lt = serde_json::to_value(&lt).unwrap().to_string();
        let ltv = serde_json::json!({"lt": 1.1}).to_string();
        assert_eq!(lt, ltv);

        let lte = RangeCase::Lte(serde_json::json!("abc"));
        let lte = serde_json::to_value(&lte).unwrap().to_string();
        let ltev = serde_json::json!({"lte": "abc"}).to_string();
        assert_eq!(lte, ltev);
    }

    #[test]
    fn clause_case() {
        let r = ClauseCase::Range("t-id", RangeCase::Gt(serde_json::json!(1)));
        let r = serde_json::to_value(&r).unwrap().to_string();
        let rv = serde_json::json!({"range": {"t-id": {"gt": 1}}}).to_string();
        assert_eq!(r, rv);

        let m = ClauseCase::Match("t-id", serde_json::json!(1));
        let m = serde_json::to_value(&m).unwrap().to_string();
        let mv = serde_json::json!({"match": {"t-id": 1}}).to_string();
        assert_eq!(m, mv);
    }

    #[test]
    fn query() {
        let q = Query {
            and: Some(vec![
                Clause::Clause(ClauseCase::Range(
                    "t-id",
                    RangeCase::Gt(serde_json::json!(1)),
                )),
                Clause::Query(Query {
                    and: Some(vec![Clause::Clause(ClauseCase::Range(
                        "t-id",
                        RangeCase::Lt(serde_json::json!(1)),
                    ))]),
                    not: None,
                    or: Some(vec![Clause::Clause(ClauseCase::Match(
                        "t-id",
                        serde_json::json!("qians"),
                    ))]),
                }),
            ]),
            not: None,
            or: None,
        };

        let q = serde_json::to_value(&q).unwrap().to_string();
        let qv = serde_json::json!({
                "and": [
                    {"range": {"t-id": {"gt": 1}}},
                    {"query": {
                        "and": [
                            {"range": {"t-id": {"lt": 1}}}
                        ],
                        "or": [
                            {"match": {"t-id": "qians"}}
                        ]
                    }}
                ]
            }
        )
        .to_string();
        assert_eq!(q, qv);
    }
}
