use serde_derive::Deserialize;

use crate::error;

#[derive(Deserialize)]
pub struct Response {
    error: String,
    data: serde_json::Value,

    #[serde(default)]
    count: u32,
}

impl Response {
    pub fn into<T>(self) -> Result<T, error::Error>
    where
        T: serde::de::DeserializeOwned,
    {
        if self.error.is_empty() {
            serde_json::from_value(self.data)
                .map_err(|e| error::Error::ResponseDeserializeErr(e.to_string()))
        } else {
            Err(error::Error::ResponseErr(self.error))
        }
    }

    pub fn count(&self) -> u32 {
        self.count
    }
}

#[cfg(test)]
mod test {
    use crate::Response;

    #[test]
    fn into() {
        #[derive(serde_derive::Deserialize, Clone, Debug, PartialEq, Eq)]
        struct A {
            a: i32,
            b: String,
            c: bool,
            d: Option<String>,
        }

        let a = A {
            a: 1,
            b: String::from("1"),
            c: true,
            d: Some(String::from("1")),
        };
        let v = vec![a.clone(), a.clone(), a.clone()];

        let json = serde_json::json!({
            "error": "",
            "count": 3,
            "data": [
                {
                    "a": 1,
                    "b": "1",
                    "c": true,
                    "d": "1",
                },
                {
                    "a": 1,
                    "b": "1",
                    "c": true,
                    "d": "1",
                },
                {
                    "a": 1,
                    "b": "1",
                    "c": true,
                    "d": "1",
                }
            ]
        })
        .to_string();
        let res: Response = serde_json::from_str(&json).unwrap();
        let res: Vec<A> = res.into().unwrap();

        assert_eq!(v, res)
    }
}
