use super::*;

pub struct Project<'p> {
    pub(super) project: &'p str,
}

impl<'p> Project<'p> {
    pub fn table<'t, 'a: 't>(self, table: &'a str) -> Table<'p, 't> {
        Table {
            project: self.project,
            table,
        }
    }
}

pub struct Table<'p, 't> {
    project: &'p str,
    table: &'t str,
}

impl<'p, 't> Table<'p, 't> {
    pub fn create(self) -> CreateBuilder<'p, 't> {
        CreateBuilder {
            project: self.project,
            table: self.table,
            data: Vec::new(),
        }
    }

    pub fn get<'o1, 'o2>(self, id: &'o1 str) -> GetBuilder<'p, 't, 'o1, 'o2> {
        GetBuilder {
            project: self.project,
            table: self.table,
            id,
            fields: Vec::new(),
        }
    }

    pub fn set<'o1>(self, id: &'o1 str) -> SetBuilder<'p, 't, 'o1> {
        SetBuilder {
            project: self.project,
            table: self.table,
            id,
        }
    }

    pub fn query<'o1>(self, query: query::Query<'o1>) -> QueryBuilder<'p, 't, 'o1> {
        QueryBuilder {
            project: self.project,
            table: self.table,
            conditions: conditions::Conditions::from_query(query),
        }
    }

    pub fn update_by_query<'o1>(
        self,
        query: query::Query<'o1>,
    ) -> UpdateByQueryBuilder<'p, 't, 'o1> {
        UpdateByQueryBuilder {
            project: self.project,
            table: self.table,
            query,
        }
    }
}

pub struct CreateBuilder<'p, 't> {
    project: &'p str,
    table: &'t str,

    data: Vec<serde_json::Value>,
}

impl<'p, 't> CreateBuilder<'p, 't> {
    pub fn create_one(mut self, v: serde_json::Value) -> CreateBuilder<'p, 't> {
        self.data.push(v);
        self
    }

    pub fn done<'o1, 'o2>(self) -> Request<'p, 't, 'o1, 'o2> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::Create { data: self.data },
        }
    }
}

pub struct GetBuilder<'p, 't, 'o1, 'o2> {
    project: &'p str,
    table: &'t str,

    id: &'o1 str,
    fields: Vec<&'o2 str>,
}

impl<'p, 't, 'o1, 'o2> GetBuilder<'p, 't, 'o1, 'o2> {
    pub fn add_field(mut self, field: &'o2 str) -> GetBuilder<'p, 't, 'o1, 'o2> {
        self.fields.push(field);
        self
    }
    pub fn add_fields(mut self, fields: &'o2 [String]) -> GetBuilder<'p, 't, 'o1, 'o2> {
        fields.iter().for_each(|field| {
            self.fields.push(field);
        });
        self
    }

    pub fn done(self) -> Request<'p, 't, 'o1, 'o2> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::Get {
                id: self.id,
                fields: self.fields,
            },
        }
    }
}

pub struct SetBuilder<'p, 't, 'o1> {
    project: &'p str,
    table: &'t str,

    id: &'o1 str,
}

impl<'p, 't, 'o1> SetBuilder<'p, 't, 'o1> {
    pub fn done<'o2>(self, data: serde_json::Value) -> Request<'p, 't, 'o1, 'o2> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::Set { id: self.id, data },
        }
    }
}

pub struct QueryBuilder<'p, 't, 'o1> {
    project: &'p str,
    table: &'t str,

    conditions: conditions::Conditions<'o1>,
}

impl<'p, 't, 'o1> QueryBuilder<'p, 't, 'o1> {
    pub fn set_pagination(mut self, page: u32, page_size: u32) -> QueryBuilder<'p, 't, 'o1> {
        self.conditions.set_pagination(page, page_size);
        self
    }

    pub fn desc(mut self, order: &'o1 str) -> QueryBuilder<'p, 't, 'o1> {
        self.conditions.desc(order);
        self
    }

    pub fn asc(mut self, order: &'o1 str) -> QueryBuilder<'p, 't, 'o1> {
        self.conditions.asc(order);
        self
    }

    pub fn add_field(mut self, field: &'o1 str) -> QueryBuilder<'p, 't, 'o1> {
        self.conditions.add_field(field);
        self
    }
    pub fn add_fields(mut self, fields: &'o1 [String]) -> QueryBuilder<'p, 't, 'o1> {
        self.conditions.add_fields(fields);
        self
    }
    pub fn done<'o2>(self) -> Request<'p, 't, 'o1, 'o2> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::Query {
                conditions: self.conditions,
            },
        }
    }
}

pub struct UpdateByQueryBuilder<'p, 't, 'o1> {
    project: &'p str,
    table: &'t str,

    query: query::Query<'o1>,
}

impl<'p, 't, 'o1> UpdateByQueryBuilder<'p, 't, 'o1> {
    pub fn done<'o2>(self, data: serde_json::Value) -> Request<'p, 't, 'o1, 'o2> {
        Request {
            auth: 2,
            project: self.project,
            table: self.table,
            operate: operate::Operate::UpdateByQuery {
                query: self.query,
                data,
            },
        }
    }
}
