//! This file models a combined sequences.

use super::*;
use crate::sequence::renderer::RenderDisplay;
use std::convert::From;
use std::convert::TryFrom;
use std::fmt::Display;
use std::iter::Iterator;

/// A `CombinedSequence` models any sequence which is not entirely either
/// numbers, uppercase, or lowercase. It does this by combining sequences which
/// are. Each internal sequence it contains has it's own state. A
/// `CombinedSequence` gives you a method to increment and display a set of
/// sequences.
/// 
/// ```rust
/// use letter_sequence::sequence::combined::CombinedSequence;
/// use std::convert::TryFrom;
///
/// // Combined sequences will roll over when calling `.next()`.
/// let mut seq = CombinedSequence::try_from("A9Z").unwrap();
/// seq.next();
/// assert_eq!( seq.to_string(), "B0A" );
/// 
/// // When created the default is for the last element in the sequence to
/// // grow, here the first "Z" from the left turns into "AA"
/// let mut seq = CombinedSequence::try_from("Z9Z").unwrap();
/// seq.next();
/// assert_eq!( seq.to_string(), "AA0A" );
///
/// // here we the right has two characters "ZZ", the number will only be
/// // incremeneted after this point.
/// let mut seq = CombinedSequence::try_from("9ZZ").unwrap();
/// seq.next();
/// assert_eq!( seq.to_string(), "10A" );
///
/// seq.nth( 26usize.pow(2) - 1 + 25 ); // 0 based- 26 letters in the alphabet
/// assert_eq!( seq.to_string(), "10ZZ" );
///
/// // You can always create a sequence that looks right, and reset it too
/// let mut seq = CombinedSequence::try_from("9ZZ").unwrap();
/// seq.reset();
/// assert_eq!( seq.to_string(), "0A" );
/// ```
#[derive(Default, Debug, Clone)]
pub struct CombinedSequence {
	inner: Vec<Sequence>
}

impl CombinedSequence {
	fn new() -> Self {
		Self::default()
	}
	pub fn push(&mut self, seq: Sequence) {
		self.inner.push(seq)
	}
	
	/// Returns an iterator over the internal sequences
	pub fn sequences(&self) -> std::slice::Iter<Sequence> {
		self.inner.iter()
	}
	/// Returns a mutable iterator over the internal seqeunces
	pub fn sequences_mut(&mut self) -> std::slice::IterMut<Sequence> {
		self.inner.iter_mut()
	}

	pub fn reset(&mut self) {
		for seq in &mut self.sequences_mut() {
			seq.reset()
		}
	}
}

impl Iterator for CombinedSequence {
	type Item = CombinedSequence;

	fn next(&mut self) -> Option<CombinedSequence> {
		for i in &mut self.sequences_mut() {
			if let Some(_) = i.next() {
				return Some(self.clone())
			}
			else {
				i.rollover();
			}
		}
		return None
	}
}

impl Display for CombinedSequence {
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		for i in self.sequences().rev() {
			write!(f, "{}", i )?
		}
		Ok(())
	}
}

impl From<Vec<Sequence>> for CombinedSequence {
	fn from(vec: Vec<Sequence>) -> Self {
		Self { inner: vec }
	}
}

impl TryFrom<&str> for CombinedSequence {
	type Error = SequenceError;

	fn try_from(input: &str) -> Result<CombinedSequence, Self::Error> {
		let mut input : &mut str = &mut input.to_owned();
		if ! input.is_ascii() {
			return Err(SequenceError::InvalidString);
		}

		let mut splits: Vec<(usize, RenderDisplay)> = Vec::new();
		let mut iter = input.char_indices();
		while let Some((i, val)) = iter.next() {
			let val = RenderDisplay::try_from(val)?;
			match splits.last() {
				// current character is rendered differently from last character
				Some((_, lastval)) if *lastval != val => splits.push( (i,val) ),
				// first character seen
				None => splits.push( (i,val) ),
				_ => ()
			}
		}

		let mut seqs = CombinedSequence::new();
		while let Some((idx, _)) = splits.pop() {
			let (init, last) = input.split_at_mut(idx);
			input = &mut(*init);
			let mut seq = SequenceBuilder::try_from(&*last).unwrap();
			if ! splits.is_empty() {
				let length = last.len();
				seq = seq.max_render_capacity(length as u8);
			}
			let seq = seq.build()?;
			seqs.push( seq );
		}
		Ok(seqs)
	}
}
