pub mod helper;
pub mod builder;
pub mod renderer;
pub mod combined;

use std::convert::Into;

use crate::sequence::renderer::*;
use std::fmt;
use super::*;

use std::ops::{AddAssign, SubAssign};

#[derive(Clone, Debug, PartialEq, Eq)]
pub struct Sequence {
	inner: u64,
	last: Option<u64>,
	length: Option<u8>,
	renderer: SeqRenderer,
	step: u32
}

impl Default for Sequence {
	fn default() -> Self {
		Self {
			inner: 0,
			last: None,
			length: None,
			renderer: Default::default(),
			step: 1
		}
	}
}

impl Iterator for Sequence {
	type Item = Self;
	fn next(&mut self) -> Option<Self::Item> {
		let cur = self.inner.clone();
		self.inc().ok().and_then(|_| Some(self.clone()))
	}
}

impl fmt::Display for Sequence {
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> Result<(), fmt::Error> {
		write!(f, "{}", self.renderer.render( self.inner ).unwrap() )
	}
}

/// ```rust
/// use letter_sequence::sequence::Sequence;
/// 
/// let mut seq = Sequence::new(42).unwrap();
///
/// use std::ops::{AddAssign, SubAssign};
/// // += and -= for unsigned int
/// seq += 10u64;
/// assert_eq!( seq.to_string(), "52" );
/// seq -= 5u32;
/// assert_eq!( seq.to_string(), "47" );
///
/// use std::convert::From;
/// use std::convert::Into;
/// let rhs = Sequence::new(10).unwrap();
/// seq += rhs;
/// assert_eq!( seq.to_string(), "57" );
/// ```
impl Sequence {


	/// Create a new instance setting the internal sequence at the start
	pub fn new(inner: u64) -> Result<Self, SequenceError> {
		let mut seq = Self::default();
		seq.set(inner)?;
		Ok(seq)
	}
	
	/// Create a new instance, as above, but do not check for violations of
	/// display
	fn new_unchecked(inner: u64) -> Self {
		Self { inner, .. Self::default() }
	}

	pub fn inc(&mut self) -> Result<(), SequenceError> {
		Sequence::add(self, self.step as u64)
	}

	/// Reset the sequence to 0
	pub fn reset(&mut self) {
		self.inner = Default::default();
	}
	
	/// Add rhs to sequence
	pub fn add(&mut self, rhs: u64) -> Result<(), SequenceError> {
		if let Some(limit) = self.renderer.max() {
			if limit < self.inner + rhs {
				return Err(SequenceError::OutOfRange)
			}
		}
		if let Some(last) = self.last {
			if last < self.inner + rhs {
				return Err(SequenceError::OutOfRange)
			}
		}
		self.inner += rhs;
		Ok(())
	}
	
	pub fn sub(&mut self, rhs: u64) -> () {
		self.inner -= rhs;
	}

	fn rollover(&mut self) -> () {
		self.inner = Default::default();
	}

	pub fn set(&mut self, inner: u64) -> Result<(), SequenceError> {
		if let Some(limit) = self.renderer.max() {
			if inner > limit {
				return Err(SequenceError::OutOfRange)
			}
		}
		self.inner = inner;
		Ok(())
	}
}

impl<T: Into<u64>> AddAssign<T> for Sequence {
	fn add_assign(&mut self, rhs: T) {
		Sequence::add(self, rhs.into()).unwrap();
	}
}

impl<T: Into<u64>> SubAssign<T> for Sequence {
	fn sub_assign(&mut self, rhs: T) {
		Sequence::sub(self, rhs.into());
	}
}

impl From<Sequence> for u64 {
	fn from(seq: Sequence)-> u64 { return seq.inner }
}


#[cfg(test)]
mod test {
	use super::*;

	#[test]
	fn default() {
		assert_eq!( Sequence::default().to_string(), "0" );
	}

	#[test]
	fn new_unchecked() {
		assert_eq!( Sequence::new_unchecked(u64::MAX).to_string(), u64::MAX.to_string() );
		let seq = SequenceBuilder::new(u64::MAX).build().unwrap();
		assert_eq!( seq.to_string(), u64::MAX.to_string() );
	}
	
	#[test]
	fn num() {
		for i in 0..123_456 {
			let seq = Sequence::new(i).unwrap();
			assert_eq!( seq.to_string(), i.to_string() );
		}
	}

	#[test]
	fn lower() {
		for i in 0..100 {
			let seq = SequenceBuilder::new(i).lower().build().unwrap();
			assert_eq!(
				seq.to_string().matches( |c: char| {
					c.is_numeric() || c.is_whitespace() || c.is_control()
				} ).into_iter().collect::<Vec<&str>>()
				, Vec::<&str>::new()
			);
			assert_eq!(
				seq.to_string(),
				seq.to_string().to_ascii_lowercase(),
			)
		}
	}
	
	#[test]
	fn upper() {
		for i in 0..100 {
			let seq = SequenceBuilder::new(i).upper().build().unwrap();
			assert_eq!(
				seq.to_string().matches( |c: char| {
					c.is_numeric() || c.is_whitespace() || c.is_control()
				} ).into_iter().collect::<Vec<&str>>()
				, Vec::<&str>::new()
			);
			assert_eq!(
				seq.to_string(),
				seq.to_string().to_ascii_uppercase(),
			)
		}
	}

	//inner: 1 * 26u64.pow(2) + 1 * 26u64.pow(1)  + 0 * 26u64.pow(0),
	#[test]
	fn rollover() {
		use std::convert::TryFrom;
		let mut seq = SequenceBuilder::try_from("Z").unwrap().build().unwrap();
		assert_eq!( seq.next().unwrap().to_string(), "AA" );
		assert_eq!( seq.next().unwrap().to_string(), "AB" );
		
		let mut seq = SequenceBuilder::try_from("z").unwrap().build().unwrap();
		assert_eq!( seq.next().unwrap().to_string(), "aa" );
		assert_eq!( seq.next().unwrap().to_string(), "ab" );
		
		let mut seq = SequenceBuilder::try_from("9").unwrap().build().unwrap();
		assert_eq!( seq.next().unwrap().to_string(), (10).to_string() );
		assert_eq!( seq.next().unwrap().to_string(), (11).to_string() );
	}

}
