#![allow(dead_code, warning)]
use std::path::PathBuf;
use structopt::StructOpt;

#[derive(Debug, Default)]
struct PositionalArgs<'A> {
	first: Option<&'A str>,
	increment: Option<&'A str>,
	last: &'A str
}

use std::ffi::OsStr;
use std::fmt::Debug;
macro_rules! parse_u64 {
	($var:ident) => {
		Some(
			$var
			.as_ref()
			.parse::<u64>()
			.expect(  &format!("Could not parse '{:?}'", stringify!($var))  )
		)
	};
}

use std::convert::TryFrom;
impl<'A, T> TryFrom<&'A [T]> for PositionalArgs<'A>
where T: 'A + std::fmt::Display + Debug + AsRef<str>
{
	type Error = &'static str;
	fn try_from(mut vec: &'A [T]) -> Result<Self, Self::Error> {
		println!("ARGUMENT: {:?}", vec);

		if vec.len() > 3 {
			panic!("Too many positional arguments")
		}

		let o = match (vec.get(0), vec.get(1), vec.get(2)) {
			( Some(last), None, None ) => {
				Self { last: last.as_ref(), .. Default::default() }
			},
			( Some(first), Some(last), None ) => {
				Self {
					last: last.as_ref(),
  				first: Some(first.as_ref()),
					..Default::default()
				}
			},
			( Some(first), Some(increment), Some(last) ) => {
				Self {
					last: last.as_ref(),
					first: Some(first.as_ref()),
					increment: Some(increment.as_ref()),
					..Default::default()
				}
			},
			_ => panic!("More than one positional argument is required")
		};
		Ok(o)
	}
}


#[structopt(verbatim_doc_comment)]
#[derive(Debug, StructOpt)]
#[structopt(name = "seq")]
/// Usage: seq [OPTION]... LAST
///   or:  seq [OPTION]... FIRST LAST
///   or:  seq [OPTION]... FIRST INCREMENT LAST
///
/// Print numbers from FIRST to LAST, in steps of INCREMENT.
///
/// If FIRST or INCREMENT is omitted, it defaults to 1.  That is, an
/// omitted INCREMENT defaults to 1 even when LAST is smaller than FIRST.
/// The sequence of numbers ends when the sum of the current number and
/// INCREMENT would become greater than LAST.
/// FIRST, INCREMENT, and LAST are interpreted as floating point values.
/// INCREMENT is usually positive if FIRST is smaller than LAST, and
/// INCREMENT is usually negative if FIRST is greater than LAST.
/// INCREMENT must not be 0; none of FIRST, INCREMENT and LAST may be NaN.
/// FORMAT must be suitable for printing one argument of type 'double';
/// it defaults to %.PRECf if FIRST, INCREMENT, and LAST are all fixed point
/// decimal numbers with maximum precision PREC, and to %g otherwise.
struct Opt {
  /// use printf style floating-point FORMAT
  #[structopt(short, long)]
  format: Option<String>,

	/// use STRING to separate numbers (default: \n)
  #[structopt(short, long)]
  seperator: Option<String>,

	/// equalize width by padding with leading zeroes
  #[structopt(short = "w", long)]
  equal_width: Option<bool>,
	
	#[structopt(required=true)]
	args: Vec<String>,
}

fn main() {	
	let mut opts = Opt::from_args();
	let pos: Vec<String> = std::mem::take(&mut opts.args);
	let pos: Vec<&str> = pos.iter().map(AsRef::as_ref).collect();	
	let pos: &[&str] = pos.as_slice();
	let pos = PositionalArgs::try_from(pos).unwrap();

	use letter_sequence::SequenceBuilder;
	let mut seq = SequenceBuilder::try_from(pos.first.unwrap()).unwrap()
		.set_last(42)
		.build()
		.unwrap();

	while let Some(e) = seq.next() {
		println!("{}", e);
	}
}
