//! This file models a combined sequences.

use super::*;
use crate::sequence::Sequence;
use crate::sequence::builder::SequenceBuilder;
use crate::sequence::renderer::RenderDisplay;
use std::convert::From;
use std::convert::TryFrom;

#[derive(Default, Debug, Clone)]
pub struct CombinedSequence {
	inner: Vec<Sequence>
}

impl From<Vec<Sequence>> for CombinedSequence {
	fn from(vec: Vec<Sequence>) -> Self {
		Self { inner: vec }
	}
}

impl CombinedSequence {
	pub fn new() -> Self {
		Self::default()
	}
	pub fn push(&mut self, seq: Sequence) {
		self.inner.push(seq)
	}
}

use std::fmt::Display;

impl Display for CombinedSequence {
	fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
		for i in self.inner.iter().rev() {
			write!(f, "{}", i )?
		}
		Ok(())
	}
}

impl TryFrom<&str> for CombinedSequence {
	type Error = SequenceError;


	/// ```rust
	/// use letter_sequence::sequence::combined::CombinedSequence;
	/// use std::convert::TryFrom;
	///
	/// let seq = CombinedSequence::try_from("A0ZA").unwrap();
	/// assert_eq!( seq.to_string(), "A0ZA" )
	/// ```
	fn try_from(input: &str) -> Result<CombinedSequence, Self::Error> {
		let mut input : &mut str = &mut input.to_owned();
		if ! input.is_ascii() {
			return Err(SequenceError::InvalidString);
		}

		let mut splits: Vec<(usize, RenderDisplay)> = Vec::new();
		let mut iter = input.char_indices();
		while let Some((i, val)) = iter.next() {
			let val = RenderDisplay::try_from(val)?;
			match splits.last() {
				Some((_, lastval)) if *lastval != val => splits.push( (i,val) ),
				None => splits.push( (i,val) ),
				_ => ()
			}
		}
		println!("SPLITS {:#?}", splits);

		let mut seqs = CombinedSequence::new();
		while let Some((idx, rd)) = splits.pop() {
			let (init, last) = input.split_at_mut(idx);
			input = &mut(*init);
			println!( "For idx {} got std [{}]", idx, &*last);
			seqs.push( SequenceBuilder::try_from(&*last).unwrap().build()? );
		}
		eprintln!("SPLITS {:#?}", seqs);
		Ok(seqs)
	}
}
