use super::*;
use crate::sequence::helper::*;
use std::convert::From;
use std::convert::TryFrom;

/// ```rust
/// use letter_sequence::sequence::builder;
/// use std::convert::TryFrom;
///
/// let mut seq = builder::SequenceBuilder::try_from("Z")
///   .unwrap()
/// 	.upper()
/// 	.build()
/// 	.unwrap();
/// assert_eq!( seq.to_string(), "Z" );
/// seq.next();
/// assert_eq!( seq.to_string(), "AA" );
/// ```
///
/// This time with .max_chars() set, and lowercase
/// ```rust
/// use letter_sequence::sequence::builder;
/// use std::convert::TryFrom;
///
/// let mut seq = builder::SequenceBuilder::try_from('y')
///   .unwrap()
/// 	.lower()
/// 	.max_chars(1)
/// 	.build()
/// 	.unwrap();
/// assert_eq!( seq.next().unwrap().to_string(), "z"  );
/// assert_eq!( seq.to_string(),                 "z"  );
/// // Without .max_chars(1) this would reutrn "aa"
/// assert_eq!( seq.next(),                      None );
/// ```
///
/// Also boring number sequence
/// ```rust
/// use letter_sequence::sequence::builder;
/// use std::convert::TryFrom;
///
/// let mut seq = builder::SequenceBuilder::from(102)
/// 	.numeric()
/// 	.build()
/// 	.unwrap();
/// assert_eq!( seq.next().unwrap().to_string(), "103"  );
/// assert_eq!( seq.to_string(),                 "103"  );
/// ```
pub struct SequenceBuilder {
	start: u64,
	display: RenderDisplay,
	bound: Option<u8>,
}

impl Default for SequenceBuilder {
	fn default() -> Self {
		Self { start: 0, bound: None, display: RenderDisplay::Upper }
	}
}

impl SequenceBuilder {
	/// Creates a new default Sequence starting at `start` (0-base)
	/// ```
	/// use letter_sequence::sequence::builder;
	/// let seq = builder::SequenceBuilder::new(2).build().unwrap();
	/// assert_eq!( seq.to_string(), "C" );
	/// ```
	pub fn new(start: u64) -> Self {
		Self { start, ..Default::default() }
	}

	pub fn new_from_string(start: &str) -> () {
		todo!("Process from a string, create sequence that represents state")
	}

	/// Set a limit on the sequence based on the chacters in rendered output, default: unlimited
	pub fn max_chars(mut self, chars: u8) -> Self {
		self.bound = Some(chars);
		self
	}
	/// Render the sequence in lowercase
	pub fn lower(mut self) -> Self {
		self.display = RenderDisplay::Lower;
		self
	}
	/// Render the sequence in uppercase, default
	pub fn upper(mut self) -> Self {
		self.display = RenderDisplay::Upper;
		self
	}
	/// Render the sequence in digits
	pub fn numeric(mut self) -> Self {
		self.display = RenderDisplay::Numeric;
		self
	}
	pub fn build(self) -> Result<Sequence, SequenceError> {
		let renderer = SeqRenderer {
			display: self.display,
			bound: self.bound,
			.. Default::default()
		};
		let mut seq = Sequence {
			inner: 0,
			renderer,
			.. Default::default()
		};
		seq.set(self.start)?;
		Ok(seq)
	}
}

impl From<u64> for SequenceBuilder {
	fn from(value: u64) -> Self {
		Self::new(value)
	}
}

impl TryFrom<&str> for SequenceBuilder {
	type Error = SequenceError;
	fn try_from(value: &str) -> Result<Self, Self::Error> {
		Ok(Self::new(string_to_int(value)?))
	}
}

impl TryFrom<char> for SequenceBuilder {
	type Error = SequenceError;
	fn try_from(value: char) -> Result<Self, Self::Error> {
		Ok(Self::new(char_to_int(value)? as u64))
	}
}

#[cfg(test)]
mod test {
	use super::*;

	/// Test violations of the bounds system, ensure we return an error
	/// Test that we can increment past respective .max_numeric and .max_alpha
	#[test]
	fn bounds() {
		// Test link between max_alpha(chars) and the OutOfRange
		for limit in 1..10 {
			let max = crate::sequence::renderer::max_alpha(limit);
			let mut seq = SequenceBuilder::new(max).max_chars(limit).build().unwrap();
			let render_as = "Z".repeat(limit as usize);
			assert_eq!( seq.to_string(), render_as );
			assert_eq!(
				seq.inc().unwrap_err(),
				SequenceError::OutOfRange,
			);
			assert_eq!( seq.next(), None )
		}

		// Test max_numeric(chars) + 1 throws OutOfRange
		for limit in 1..10 {
			let max = crate::sequence::renderer::max_numeric(limit);
			let mut seq = SequenceBuilder::new(max).numeric().max_chars(limit).build().unwrap();
			let render_as = "9".repeat(limit as usize);
			assert_eq!( seq.to_string(), render_as, "Render {} as {}", max, render_as );
			assert_eq!(
				seq.inc().unwrap_err(),
				SequenceError::OutOfRange,
				"Throw error with numeric char limit of {} which can only represent {}",
					limit,
					max
			);
			assert_eq!( seq.next(), None )
		}
	}

	#[test]
	fn simple() {
		for c in 'A'..'Z' {
			let seq = SequenceBuilder::new(c as u64 - 'A' as u64).upper().build().unwrap();
			assert_eq!(
				seq.to_string(),
				c.to_string(),
			);

			let seq = SequenceBuilder::new(c as u64 - 'A' as u64).lower().build().unwrap();
			assert_eq!(
				seq.to_string(),
				c.to_string().to_lowercase(),
			);
		}
		for i in 1..123_456 {
			assert_eq!(
				SequenceBuilder::new(i).numeric().build().unwrap().to_string(),
				i.to_string()
			);
		}
	}

	#[test]
	fn rollover() {
		for i in A..Z {

			// Easy check for A[A-Z] after Z
			let seq = SequenceBuilder::new(Z as u64).upper().build().unwrap();
			let result = format!("A{}", std::char::from_u32('A' as u32 + i).unwrap());
			assert_eq!(
				seq.skip(i as usize).next().unwrap().to_string(),
				result
			);

			
			let mut seq = SequenceBuilder::new(Z as u64).upper().build().unwrap();
			assert_eq!(
				seq.nth(i as usize).unwrap().to_string(),
				result,
				".nth({})", i
			);

			let seq = SequenceBuilder::new(Z as u64 + 1 + i as u64).lower().build().unwrap();
			assert_eq!(
				seq.to_string(),
				result.to_lowercase()
			);
			
			let mut seq = SequenceBuilder::new(A as u64).upper().build().unwrap();
			for len in 1..5 {
				assert_eq!(
					seq.nth(26usize.pow(len as u32) - 1).unwrap().to_string(),
					"A".repeat(len + 1),
				);
			}
			
			let mut seq = SequenceBuilder::new(Z as u64).upper().build().unwrap();
			for len in 2..5 {
				assert_eq!(
					seq.nth(26usize.pow(len as u32) - 1).unwrap().to_string(),
					"Z".repeat(len),
				);
			}
		}
	}

}
