use super::*;
use crate::sequence::helper::*;

#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub(crate) enum RenderDisplay {
	Upper,
	Lower,
	Numeric,
}
impl Default for RenderDisplay {
	fn default() -> Self { RenderDisplay::Numeric }
}

#[derive(Default, Clone, Copy, Debug, PartialEq, Eq)]
pub(crate) struct SeqRenderer {
	pub(crate) display: RenderDisplay,
	pub(crate) bound: Option<u8>,
}

impl SeqRenderer {
	pub fn render(&self, x: u64) -> Result<String, SequenceError> {
		let result = match self.display {
			RenderDisplay::Upper => int_to_string(x)?,
			RenderDisplay::Lower => {
				let mut display = int_to_string(x)?;
				display.make_ascii_lowercase();
				display.to_string()
			}
			RenderDisplay::Numeric => x.to_string()
		};
		// Ensure that the result doesn't exceed the bound
		match self.bound {
			Some(c) if result.len() > c as usize => Err(SequenceError::OutOfRange),
			Some(_) => Ok(result),
			None => Ok(result),
		}
	}
	pub fn max(&self) -> Option<u64> {
		if let Some(char_limit) = self.bound {
			match self.display {
				RenderDisplay::Upper | RenderDisplay::Lower => Some(max_alpha(char_limit)),
				RenderDisplay::Numeric => Some(max_numeric(char_limit)),
			}
		}
		else {
			None
		}
	}
}

/// Returns the maximum int that can be represented with `chars` alphabetic
/// characters
/// ```
/// use letter_sequence::sequence::renderer;
/// assert_eq!( renderer::max_alpha(1u8), 25 );
/// assert_eq!( renderer::max_alpha(2u8), 26u64.pow(2) + 25 );
/// ```
pub fn max_alpha(chars: u8) -> u64 {
	const BASE : u64 = 26;
	let mut max: u64 = 0;
	for x in 0..chars {
		max += (1 + Z) as u64 * BASE.pow(x as u32)
	}
	max -= 1;
	max
}

/// Finds the maximum int that can be represented with `chars` digits
/// ```
/// use letter_sequence::sequence::renderer;
/// // Simple string comparison for digits.
/// # for i in 1..15 {
/// 	assert_eq!( renderer::max_numeric(i).to_string(), "9".repeat(i as usize) );
/// # }
/// ```
pub fn max_numeric(chars: u8) -> u64 {
	10u64.pow(chars as u32) - 1
}
