#[macro_use]
extern crate diesel;

use diesel::pg::PgConnection;
use diesel::Connection;
use diesel::QueryDsl;
use length_aware_paginator::{Paginate, Response};
use serde::{Deserialize, Serialize};

/// Get the database connection
/// *panics* if no DATABASE_URL is defined in the env or if the db is unreachable
fn get_connection() -> PgConnection {
    let database_url =
        dotenv::var("DATABASE_URL").expect("You have to provide DATABASE_URL to run tests");

    PgConnection::establish(&database_url)
        .unwrap_or_else(|_| panic!("Error connecting to {}", &database_url))
}

// schema.rs : autogenerated by diesel after running migration
table! {
    users (id) {
        id -> Int4,
        email -> Varchar,
        first_name -> Varchar,
        last_name -> Varchar,
        password -> Varchar,
    }
}

// user.rs : your model for the table represented in schema.rs
#[derive(Queryable, Deserialize, Serialize)]
pub struct User {
    id: i32,
    email: String,
    first_name: String,
    last_name: String,
    password: String,
}

#[test]
fn test_orm_query_pagination() {
    let mut connection = get_connection();

    let response: Response<User> = users::table
        .into_boxed()
        .page(Some(1))
        .per_page(Some(10))
        .paginate::<User>(&mut connection)
        .unwrap();

    assert_eq!(response.page, 1);
    assert_eq!(response.per_page, 10);
    assert_eq!(response.total, 15);
    assert_eq!(response.last_page, 2);
    assert_eq!(response.data.len(), 10);
}

#[test]
fn test_orm_query_pagination_per_page_first() {
    let mut connection = get_connection();

    let response: Response<User> = users::table
        .into_boxed()
        .per_page(Some(10))
        .page(Some(1))
        .paginate::<User>(&mut connection)
        .unwrap();

    assert_eq!(response.page, 1);
    assert_eq!(response.per_page, 10);
    assert_eq!(response.total, 15);
    assert_eq!(response.last_page, 2);
    assert_eq!(response.data.len(), 10);
}

#[test]
fn test_orm_query_pagination_second_page() {
    let mut connection = get_connection();

    let response: Response<User> = users::table
        .into_boxed()
        .page(Some(2))
        .per_page(Some(10))
        .paginate::<User>(&mut connection)
        .unwrap();

    assert_eq!(response.page, 2);
    assert_eq!(response.per_page, 10);
    assert_eq!(response.total, 15);
    assert_eq!(response.last_page, 2);
    assert_eq!(response.data.len(), 5);
}
