use std::fmt;
use std::fs;
use std::io::prelude::*;
use std::path::Path;
use walkdir::WalkDir;

pub struct BadLine {
    path: String,
    line: usize,
    len: usize,
    limit: usize,
}

impl fmt::Display for BadLine {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(
            f,
            "   - {}:{}:{}, ({}/{})",
            self.path, self.line, self.limit, self.len, self.limit
        )
    }
}

pub fn run(
    paths: Vec<&str>,
    line_length: usize,
    comment_length: usize,
    excludes: Vec<&str>,
) -> Result<Vec<BadLine>, String> {
    let mut bad_lines: Vec<BadLine> = vec![];

    for path in paths {
        'main: for entry in WalkDir::new(path) {
            let entry = entry.map_err(|e| format!("Error walking through directories: {}", e))?;
            let path = entry.path();
            let ext = match path.extension() {
                Some(e) => match e.to_str() {
                    Some(s) => s,
                    None => continue 'main,
                },
                None => continue 'main,
            };
            if path.is_dir() || !(ext.ends_with("pyw") || ext.ends_with("py")) {
                continue;
            }
            for e in &excludes {
                if match path.to_str() {
                    None => continue 'main,
                    Some(s) => s.contains(e),
                } {
                    continue 'main;
                }
            }
            bad_lines = check_file(path, bad_lines, line_length, comment_length)?;
        }
    }

    Ok(bad_lines)
}

fn check_file(
    path: &Path,
    mut bad_lines: Vec<BadLine>,
    line_length: usize,
    comment_length: usize,
) -> Result<Vec<BadLine>, String> {
    let display = path.display();

    let mut file =
        fs::File::open(&path).map_err(|e| format!("Couldn't open {}: {}", display, e))?;

    let mut s = "".to_string();
    file.read_to_string(&mut s)
        .map_err(|e| format!("Couldn't read {}: {}", display, e))?;

    let mut in_docs = false;
    let mut in_license = false;

    for (i, line) in s.lines().enumerate() {
        let ls = line.trim_start();
        let rs = line.trim_end();
        if in_license {
            if ls.starts_with('#') {
                continue;
            }

            in_license = false;
        }
        if ls.starts_with("\"\"\"") {
            in_docs = true;
        }

        let limit = {
            if in_docs || ls.starts_with('#') {
                comment_length
            } else {
                line_length
            }
        };
        let len = line.chars().count();
        if len > limit {
            bad_lines.push(BadLine {
                path: format!("{}", display),
                line: i + 1,
                len,
                limit,
            })
        }

        if rs.ends_with("\"\"\"") {
            in_docs = false;
        }
    }

    Ok(bad_lines)
}
