use clap::{crate_name, crate_version, App, Arg};
use len9_lib::run;
use std::process;

fn create_app() -> App<'static, 'static> {
    App::new(crate_name!())
        .version(crate_version!())
        .about("len8 rewrite in rust")
        .arg(
            Arg::with_name("PATHS")
                .required(true)
                .multiple(true)
                .help("The paths len9 should check"),
        )
        .arg(
            Arg::with_name("line-length")
                .long("line-length")
                .short("l")
                .takes_value(true)
                .value_name("LINE_LENGTH")
                .default_value("79")
                .help("The max line length of code lines."),
        )
        .arg(
            Arg::with_name("comment-length")
                .long("comment-length")
                .short("c")
                .takes_value(true)
                .value_name("COMMENT_LENGTH")
                .default_value("72")
                .help("The max line length of comment lines."),
        )
        .arg(
            Arg::with_name("excludes")
                .long("excludes")
                .short("x")
                .require_delimiter(true)
                .multiple(true)
                .takes_value(true)
                .value_name("EXCLUDES")
                .help("A comma separated list of paths to exclude (.venv, .nox and venv are already excluded)."),
        )
}

fn main() {
    let matches = create_app().get_matches();

    let line_length: usize = matches
        .value_of("line-length")
        .unwrap_or_else(|| panic!("This should never happen"))
        .parse()
        .unwrap_or_else(|e| {
            eprintln!("❌ Error parsing line length: {}", e);
            process::exit(1)
        });
    let comment_length: usize = matches
        .value_of("comment-length")
        .unwrap_or_else(|| panic!("This should never happen"))
        .parse()
        .unwrap_or_else(|e| {
            eprintln!("❌ Error parsing comment length: {}", e);
            process::exit(1)
        });

    let paths: Vec<&str> = matches
        .values_of("PATHS")
        .unwrap_or_else(|| panic!("This should never happen"))
        .collect();

    let mut excludes: Vec<&str> = match matches.values_of("excludes") {
        None => vec![],
        Some(e) => e.collect(),
    };
    excludes.push("venv");
    excludes.push(".venv");
    excludes.push(".nox");

    println!("Checking for too long lines...");
    let bad_lines = run(paths, line_length, comment_length, excludes).unwrap_or_else(|s| {
        eprintln!("❌ {}", s);
        process::exit(1)
    });
    let bad_lines_len = bad_lines.len();
    if bad_lines_len > 0 {
        println!("{} line(s) are too long:", bad_lines_len);
        let mut string_list: Vec<String> = vec![];
        for bad_line in bad_lines {
            string_list.push(format!("{}", bad_line))
        }
        println! {"{}", string_list.join("\n")}
    } else {
        println!("🎉 No too long lines found!")
    }
}
