use crate::constants::ServiceMap;
use crate::records::{ServiceEnum, ServiceRecord};
use std::error::Error;
use std::io::{ErrorKind, Read, Write};
use std::net::{SocketAddr, TcpStream};
use std::time::Duration;

pub struct Scnr {}

impl Scnr {
    /*

    connect scan

     */
    ///TCP connect scan, no timeout.
    /// address is the IPv6 or v4 address, the ports are the post to check.
    /// returns open ports as i32.
    pub fn connect_scan(address: &str, ports: Vec<i32>) -> Result<Vec<i32>, Box<dyn Error>> {
        let mut open_port_vector: Vec<i32> = vec![];

        for port in ports {
            let addr_string = format!("{}:{}", address, port);
            let server_details: SocketAddr = addr_string.parse().expect("unable to parse");

            if TcpStream::connect(&server_details).is_ok() {
                open_port_vector.push(port);
            }
        }

        Ok(open_port_vector)
    }

    /*

    connect timeout scan

     */
    ///Timeout scan is a tcp connect scan with a timeout.
    /// More suitable if there are going to be a lot of closed ports.
    /// address is IPv6 or IPv4 format, ports is ports to scan and timeout is the
    /// amount of time you are willing to wait.
    pub fn timeout_scan(
        address: &str,
        ports: Vec<i32>,
        timeout: u64,
    ) -> Result<Vec<i32>, Box<dyn Error>> {
        let mut open_port_vector: Vec<i32> = vec![];
        let duration = Duration::from_millis(timeout);

        for port in ports {
            let addr_string = format!("{}:{}", address, port);
            let server_details: SocketAddr = addr_string.parse().expect("unable to parse");

            if TcpStream::connect_timeout(&server_details, duration).is_ok() {
                open_port_vector.push(port);
            }
        }

        Ok(open_port_vector)
    }

    /**
      Props:
       address: IPv4/v6 address being scanned

       ports: port vector to scan,

       timeout: tcp stream connect timeout

       This function does a tcp connect try and then attempts to grab a banner from the
       open port.
    */

    pub fn service_scan(
        address: &str,
        ports: Vec<i32>,
        timeout: u64,
    ) -> Result<Vec<ServiceRecord>, Box<dyn Error>> {
        let mut service_vector: Vec<ServiceRecord> = vec![];
        let duration = Duration::from_millis(timeout);
        let service_map = ServiceMap::get_service_map();
        for port in ports {
            let addr_string = format!("{}:{}", address, port);
            let server_details: SocketAddr = addr_string.parse().expect("unable to parse");
            let result = TcpStream::connect_timeout(&server_details, duration);

            if let Ok(stream) = result {
                match Grabber::stream_grab(stream, 150) {
                    Ok(banner) => {
                        let service_enum_ref = service_map.get(&port).unwrap_or(&ServiceEnum::NIL);
                        let service = get_service(service_enum_ref);
                        let banner = if banner.is_empty() {
                            "nil".to_string()
                        } else {
                            parse_banner(&banner)
                        };

                        service_vector.push(ServiceRecord {
                            port,
                            banner,
                            service,
                        });
                    }
                    Err(_) => {
                        let service_enum_ref = service_map.get(&port).unwrap_or(&ServiceEnum::NIL);
                        let service = get_service(service_enum_ref);
                        service_vector.push(ServiceRecord {
                            port,
                            banner: "nil".to_string(),
                            service,
                        });
                    }
                }
            }
        }

        Ok(service_vector)
    }
}

pub struct Grabber {}

impl Grabber {
    /*

    tcp send string

     */
    ///Sends a string with a connect timeout.
    /// Address is IPv4 and IPv6 &str.
    ///msg is the message to send.
    ///timeout is time to wait to connect.
    ///Returns string, useful for poking around.
    pub fn send_string(
        address: &str,
        port: i32,
        msg: &str,
        timeout: u64,
    ) -> Result<String, Box<dyn Error>> {
        let addr_string = format!("{}:{}", address, port);
        let server_details: SocketAddr = addr_string.parse().expect("unable to parse");
        let duration = Duration::from_millis(timeout);
        let mut stream =
            TcpStream::connect_timeout(&server_details, duration).expect("could not connect");
        let mut response_shell = String::new();
        let req_bytes = msg.as_bytes();
        stream.write_all(req_bytes).expect("could not write bytes");

        stream
            .read_to_string(&mut response_shell)
            .expect("could not read string");

        Ok(response_shell)
    }

    /*

    tcp send string
    thanks polonium https://github.com/creekorful/polonium
     */
    ///does a banner grab.
    /// Address is IPv6 or v6 &str.
    /// port is i32, its the port to grab a banner.
    /// timeout is for the read/write/connect.
    pub fn banner_grab(address: &str, port: i32, timeout: u64) -> Result<String, Box<dyn Error>> {
        let addr_string = format!("{}:{}", address, port);
        let server_details: SocketAddr = addr_string.parse().expect("unable to parse");
        let duration = Duration::from_millis(timeout);
        let rw_duration = Duration::from_millis(timeout);

        let mut banner = String::new();
        let mut stream =
            TcpStream::connect_timeout(&server_details, duration).expect("could not connect");
        stream.set_read_timeout(Option::from(rw_duration))?;
        stream.set_write_timeout(Option::from(rw_duration))?;

        let result = stream.read_to_string(&mut banner);
        if result.is_ok() && !banner.is_empty() {
            return Ok(banner);
        }

        // If timeout related error happens, do not fails
        // because we may need to talk first
        let error = result.err().unwrap();
        if error.kind() != ErrorKind::WouldBlock {
            return Err(error.into());
        }

        if banner.is_empty() {
            stream.write_all("GET / HTTP/1.1\n\n".as_ref())?;
            stream.read_to_string(&mut banner)?;
        }

        Ok(banner)
    }
    /*
       Props:

       stream: the input stream being used to banner grab

       timeout: the read/write timeouts

       This is so we can grab a banner and also set a timeout without a way to handle the
       error of a timeout that might block
    */
    pub fn stream_grab(mut stream: TcpStream, timeout: u64) -> Result<String, Box<dyn Error>> {
        let rw_duration = Duration::from_millis(timeout);
        let mut banner = String::new();

        stream.set_read_timeout(Option::from(rw_duration))?;
        stream.set_write_timeout(Option::from(rw_duration))?;

        let result = stream.read_to_string(&mut banner);
        if result.is_ok() && !banner.is_empty() {
            return Ok(banner);
        }

        let error = result.err().unwrap();
        if error.kind() != ErrorKind::WouldBlock {
            return Err(error.into());
        }

        if banner.is_empty() {
            stream.write_all("HEAD / HTTP/1.1\n\n".as_ref())?;
            stream.read_to_string(&mut banner)?;
        }

        Ok(banner)
    }
}

fn get_service(service_enum: &ServiceEnum) -> ServiceEnum {
    match service_enum {
        ServiceEnum::CUPS => ServiceEnum::CUPS,
        ServiceEnum::SMTPS => ServiceEnum::SMTPS,
        ServiceEnum::SSH => ServiceEnum::SSH,
        ServiceEnum::SMTP => ServiceEnum::SMTP,
        ServiceEnum::SNMP => ServiceEnum::SNMP,
        ServiceEnum::HTTPS => ServiceEnum::HTTPS,
        ServiceEnum::HTTP => ServiceEnum::HTTP,
        ServiceEnum::DNS => ServiceEnum::DNS,
        ServiceEnum::TELNET => ServiceEnum::TELNET,
        ServiceEnum::FTP => ServiceEnum::FTP,
        ServiceEnum::IMAP => ServiceEnum::IMAP,
        ServiceEnum::IMAPS => ServiceEnum::IMAPS,
        ServiceEnum::IRC => ServiceEnum::IRC,
        ServiceEnum::NNTP => ServiceEnum::NNTP,
        ServiceEnum::NTP => ServiceEnum::NTP,
        ServiceEnum::POP3 => ServiceEnum::POP3,
        _ => ServiceEnum::NIL,
    }
}

fn parse_banner(banner: &str) -> String {
    let b_vec = Vec::from_iter(banner.split(' '));
    String::from(b_vec[0])
}
