use crate::records::{BannerRecord, BasicScanRecord, ServiceScanRecord};
use crate::tcp_worker::{Grabber, Scnr};
use dns_lookup::lookup_host;
use std::error::Error;
use std::net::IpAddr;
use std::str::FromStr;

pub struct Conductor {}

impl Conductor {
    /**
    Props:

    addr: host

    msg: message to send

    port_to: port sending it to

    This is to write strings via tcp to different ports.
    */
    pub fn conduct_send_string(
        addr: &str,
        msg: &str,
        port_to: i32,
    ) -> Result<BannerRecord, Box<dyn Error>> {
        let sanitized_host = host_sanitizer(addr);
        let response = Grabber::send_string(&sanitized_host, port_to, msg, 80)?;
        let banner: BannerRecord = BannerRecord {
            host: sanitized_host,
            port: port_to,
            banner: response,
        };
        Ok(banner)
    }
    /**
    Props:

    addr: host

    port_to: port sending it to

    This attempts to grab a banner, this creates a TCP stream.
     */
    pub fn conduct_banner_grab(addr: &str, port_to: i32) -> Result<BannerRecord, Box<dyn Error>> {
        let sanitized_host = host_sanitizer(addr);
        let response = Grabber::banner_grab(&sanitized_host, port_to, 80)?;
        let banner: BannerRecord = BannerRecord {
            host: sanitized_host,
            port: port_to,
            banner: response,
        };
        Ok(banner)
    }

    /**
    Props:

    addr: host

    ports: ports being checked

    This does a basic connect scan. This doesn't have a timeout.
    Use with fewer ports, otherwise every bad connect times out long.
     */
    pub fn conduct_connect_scan(
        addr: &str,
        ports: Vec<i32>,
    ) -> Result<BasicScanRecord, Box<dyn Error>> {
        let sanitized_host = host_sanitizer(addr);
        let open_ports = Scnr::connect_scan(&sanitized_host, ports)?;
        let record = BasicScanRecord {
            host: sanitized_host,
            open_ports,
        };

        Ok(record)
    }
    /**
    Props:

    addr: host

    ports: ports to check

    timeout: time willing to wait

    This does a connect scan like conduct_connect_scan, but has a timeout as opposed to waiting.
    */
    pub fn conduct_timeout_scan(
        addr: &str,
        ports: Vec<i32>,
        timeout: u64,
    ) -> Result<BasicScanRecord, Box<dyn Error>> {
        let sanitized_host = host_sanitizer(addr);
        let open_ports = Scnr::timeout_scan(&sanitized_host, ports, timeout)?;
        let record = BasicScanRecord {
            host: sanitized_host,
            open_ports,
        };

        Ok(record)
    }

    /**
    Props:

    addr: host

    ports: ports to check

    timeout: time willing to wait

    This banner grabs all open ports.
     */
    pub fn conduct_service_scan(
        addr: &str,
        ports: Vec<i32>,
        timeout: u64,
    ) -> Result<ServiceScanRecord, Box<dyn Error>> {
        let host = host_sanitizer(addr);
        let service_records = Scnr::service_scan(&host, ports, timeout)?;

        let record = ServiceScanRecord {
            host,
            service_records,
        };

        Ok(record)
    }
    /**
    Props:

    addr: host

    This just looks up hosts.
     */
    pub fn conduct_host_lookup(addr: &str) -> Result<Vec<String>, Box<dyn Error>> {
        let ip_vecs: Vec<IpAddr> = lookup_host(addr).unwrap();
        let mut return_array: Vec<String> = vec![];

        for ip in ip_vecs {
            return_array.push(ip.to_string())
        }

        Ok(return_array)
    }
}

fn host_sanitizer(addr: &str) -> String {
    match IpAddr::from_str(addr) {
        Err(_) => {
            let ip_vecs: Vec<std::net::IpAddr> = lookup_host(addr).unwrap();

            ip_vecs[0].to_string()
        }

        Ok(IpAddr::V4(..)) => addr.to_string(),
        Ok(IpAddr::V6(..)) => addr.to_string(),
    }
}
