use anyhow::{Error, Result};
use futures::{future, stream, StreamExt, TryStream, TryStreamExt};
use lembaran::{
    stream::pagination,
    web_linking::{Link, Links, Param},
};
use serde::Deserialize;
use surf::Url;

#[derive(Deserialize, Debug)]
struct Repository {
    name: String,
}

fn repositories() -> impl TryStream<Ok = Repository, Error = Error> {
    // Create HTTP client.
    let http = surf::client();

    pagination::with_factory(move |url: Option<Url>| {
        let http = http.clone();

        async move {
            // Get URL from previous request or
            let url = match url {
                Some(x) => x,
                None => "https://api.github.com/orgs/rust-lang/repos".parse()?,
            };

            // Create request.
            let req = http
                .get(url)
                .header(
                    "user-agent",
                    format!("{}/{}", env!("CARGO_PKG_NAME"), env!("CARGO_PKG_VERSION")),
                )
                .header("accept", "application/vnd.github.v3+json")
                .send();

            // Send request.
            let mut res = req.await.unwrap();

            // Get link headers.
            let link_headers = res.header("link").unwrap();

            // Convert into links.
            let mut links = link_headers.iter().map(Into::<Links>::into).flatten();

            // Find next page URL.
            let next_link = links
                .find(|Link { params, .. }| {
                    params
                        .iter()
                        .find(|Param { name, value }| {
                            *name == "rel" && value.as_deref() == Some("next".into())
                        })
                        .is_some()
                })
                .map(|Link { uri, .. }| uri)
                .map(|x| String::from_utf8_lossy(&**x).parse::<Url>())
                .transpose()?;

            // Deserialize body.
            let data: Vec<Repository> = res.body_json().await.unwrap();

            // Stream will end when the `next_link` is `None`.
            Ok((data, next_link))
        }
    })
    .flat_map(|x: Result<Vec<_>, _>| match x {
        Ok(x) => stream::iter(x).map(|x| Ok(x)).boxed(),
        Err(x) => stream::once(future::ready(Err(x))).boxed(),
    })
}

#[async_std::main]
async fn main() -> Result<()> {
    repositories()
        .try_for_each(|Repository { name, .. }| {
            // print name to stdout
            println!("{}", name);
            future::ok(())
        })
        .await?;
    Ok(())
}
