//! Extracts URI from HTTP response implementing [RFC 8288: Web Linking].
//!
//! [RFC 8288: Web Linking]: https://www.rfc-editor.org/rfc/rfc8288.html

use bstr::{BStr, ByteSlice};
use std::borrow::Cow;

/// Construct iterator over [`Link`] from `&`[`BStr`].
pub fn from_bstr<'a>(input: impl Into<&'a BStr>) -> Links<'a> {
    let input = input.into();
    let links = read_links(input);
    links
}

#[test]
fn test_from_bstr() {
    let input = r#"<https://api.github.com/organizations/5430905/repos?page=2>; rel="next", <https://api.github.com/organizations/5430905/repos?page=6>; rel="last""#;
    let links = from_bstr(input);
    let links: Vec<_> = links.collect();
    assert_eq!(links.len(), 2);
}

#[derive(Debug)]
/// Iterator over [`Link`].
pub struct Links<'a> {
    input: &'a BStr,
}

impl<'a> Iterator for Links<'a> {
    type Item = Link<'a>;

    fn next(&mut self) -> Option<Self::Item> {
        if self.input.is_empty() {
            return None;
        }
        // B.2.X
        // Discard "," if it's the first character.
        let input = if self.input.first().map(|&x| x == b',').unwrap_or_default() {
            &self.input[1..]
        } else {
            self.input
        };
        let (link, input) = read_link(input);
        self.input = input;
        link.into()
    }
}

#[derive(Debug)]
/// Container data type for link.
pub struct Link<'a> {
    /// URI.
    pub uri: &'a BStr,

    /// Parameters.
    pub params: Vec<Param<'a>>,
}

/// Implementation of Parsing a Link Field Value from Appendinx B.2. in [RFC 8288].
///
/// Implemented as iterator [`Links`].
///
/// Operation 2. is implemented in function [`read_link`].
///
/// [RFC 8288]: https://www.rfc-editor.org/rfc/rfc8288.html#appendix-B.2
fn read_links(input: &BStr) -> Links<'_> {
    Links { input }
}

/// Implementation of Parsing a Link Field Value operation 2. from Appendinx B.2. in [RFC 8288].
///
/// [RFC 8288]: https://www.rfc-editor.org/rfc/rfc8288.html#appendix-B.2
fn read_link(input: &BStr) -> (Option<Link<'_>>, &BStr) {
    // B.2.2.1
    // Consume any leading OWS.
    let input = input.trim_start();
    // B.2.2.2
    // If the first character is not "<", return links.
    if input.first().map(|&x| x != b'<').unwrap_or_default() {
        return (None, input.into());
    }
    // B.2.2.3
    // Discard the first character ("<").
    let input = &input[1..];
    // B.2.2.4
    // Consume up to but not including the first ">" character or end of field_value and let the result be
    // target_string.
    let term = input.find_byte(b'>').unwrap_or_else(|| input.len());
    let uri = &input[..term];
    let input = &input[term..];
    // B.2.2.5
    // If the next character is not ">", return links.
    if input.first().map(|&x| x != b'>').unwrap_or_default() {
        return (None, input.into());
    }
    // B.2.2.6
    // Discard the leading ">" character.
    let input = &input[1..];
    // B.2.2.7
    // Let link_parameters be the result of Parsing Parameters (Appendix B.3) from field_value (consuming zero or
    // more characters of it).
    let (params, input) = read_params(input.into());
    // B.2.2.8
    // Let target_uri be the result of relatively resolving (as per [RFC3986], Section 5.2) target_string. Note
    // that any base URI carried in the payload body is NOT used.
    // Not implemented.
    // URI validation & parsing are deferred to user.
    //
    // B.2.2.9..=17
    // Not implemented.
    //
    (
        Link {
            uri: uri.into(),
            params,
        }
        .into(),
        input,
    )
}

#[test]
fn test_read_link() {
    let input = r#"<https://example.com>; rel="preconnect""#;
    let (link, rem) = read_link(input.into());
    let link = link.unwrap();
    assert_eq!(link.uri, "https://example.com");
    let params = link.params;
    assert_eq!(params.len(), 1);
    assert_eq!(
        params[0],
        Param {
            name: "rel".into(),
            value: Some(Cow::Borrowed("preconnect".into())),
        }
    );
    assert_eq!(rem, "");
}

#[test]
fn test_read_multiple_links() {
    let input = r#"<https://api.github.com/organizations/5430905/repos?page=2>; rel="next", <https://api.github.com/organizations/5430905/repos?page=6>; rel="last""#;
    let (link, rem) = read_link(input.into());
    let link = link.unwrap();
    assert_eq!(
        link.uri,
        "https://api.github.com/organizations/5430905/repos?page=2"
    );
    let params = link.params;
    assert_eq!(params.len(), 1);
    assert_eq!(
        params[0],
        Param {
            name: "rel".into(),
            value: Some(Cow::Borrowed("next".into())),
        }
    );
    assert_eq!(
        rem,
        r#", <https://api.github.com/organizations/5430905/repos?page=6>; rel="last""#
    );
}

/// Implementation of Parsing Parameters from Appendinx B.3. in [RFC 8288].
///
/// Operations 2.4. until 2.10. are implemented in function [`read_param`].
///
/// [RFC 8288]: https://www.rfc-editor.org/rfc/rfc8288.html#appendix-B.3
fn read_params(input: &BStr) -> (Vec<Param<'_>>, &BStr) {
    // B.3.1
    // Let parameters be an empty list.
    let mut params = Vec::new();
    //
    let mut input = input;
    let input = loop {
        // B.3.2
        // While input has content:
        if input.is_empty() {
            break input;
        }
        input = {
            // B.3.2.1
            // Consume any leading OWS.
            let input = input.trim_start();
            // B.3.2.2
            // If the first character is not ";", return parameters.
            if input.first().map(|&x| x != b';').unwrap_or_default() {
                return (params, input.into());
            }
            // B.3.2.3
            // Discard the leading ";" character.
            let input = &input[1..];
            // B.3.2.4..=B.3.2.10
            let (param, input) = read_param(input.into());
            // B.3.2.10
            // Append (parameter_name, parameter_value) to parameters.
            params.push(param);
            // B.3.2.11
            // Consume any leading OWS.
            let input = input.trim_start();
            // B.3.2.12
            // If the next character is "," or the end of input, stop processing input and return parameters.
            if input.first().map(|&x| x == b',').unwrap_or_default() || input.is_empty() {
                return (params, input.into());
            }
            input.into()
        };
    };
    (params, input)
}

#[cfg(test)]
#[test]
fn test_read_params() {
    let input = r#"; rel="previous"; title="previous chapter","#.into();
    let (params, rem) = read_params(input);
    assert_eq!(params.len(), 2);
    assert_eq!(rem, ",");
}

#[derive(PartialEq, Debug)]
/// Container data type for link parameter.
pub struct Param<'a> {
    /// Parameter name.
    pub name: &'a BStr,

    /// Parameter value.
    pub value: Option<Cow<'a, BStr>>,
}

/// Implementation of Parsing Parameters operations 2.4. until 2.10. from Appendinx B.3. in [RFC 8288].
///
/// [RFC 8288]: https://www.rfc-editor.org/rfc/rfc8288.html#appendix-B.3
fn read_param(input: &BStr) -> (Param<'_>, &BStr) {
    // B.3.2.4
    // Consume any leading OWS.
    let input = input.trim_start();
    // B.3.2.5
    // Consume up to but not including the first BWS, "=", ";", or "," character, or up to the end of input,
    // let the result be parameter_name.
    let sep = input.find_byteset(b" =;,");
    let (name, input): (&BStr, &BStr) = match sep {
        Some(sep) => {
            let name = &input[0..sep];
            let input = &input[sep..];
            (name.into(), input.into())
        }
        None => (input.into(), input[input.len()..].into()),
    };
    // B.3.2.6
    // Consume any leading BWS.
    let input = input.trim_start();
    //
    let (value, input) = if input.first().map(|&x| x == b'=').unwrap_or_default() {
        // B.3.2.7
        // If the next character is "=":
        //
        // B.3.2.7.1
        // Discard the leading "=" character.
        let input = &input[1..];
        // B.3.2.7.2
        // Consume any leading BWS.
        let input = input.trim_start();
        //
        let (value, input) = if input.first().map(|&x| x == b'"').unwrap_or_default() {
            // B.3.2.7.3
            // If the next character is DQUOTE, let parameter_value be the result of Parsing a Quoted String
            // (Appendix B.4) from input (consuming zero or more characters of it).
            let (value, input) = read_quoted_string(input.into());
            let value = value.map(|x| Cow::Owned(x.into()));
            (value, input)
        } else {
            // B.3.2.7.4
            // Else, consume the contents up to but not including the first ";" or "," character, or up to the
            // end of input, and let the results be parameter_value.
            let mut parts = input.split(|x| [b';', b','].contains(x));
            let value = parts.next().map(|x| Cow::Borrowed(x.into()));
            let input = parts.next().unwrap_or_default();
            (value, input.into())
        };
        // B.3.2.7.5
        // If the last character of parameter_name is an asterisk ("*"), decode parameter_value according to
        // [RFC8187]. Continue processing input if an unrecoverable error is encountered.
        // Not implemented.
        //
        (value, input)
    } else {
        // B.3.2.8
        // Else:
        //
        // B.3.2.8.1
        // Let parameter_value be an empty string.
        (None, input.into())
    };
    // B.3.2.9
    // Case-normalise parameter_name to lowercase.
    // Not implemented.
    //
    (Param { name, value }, input)
}

#[cfg(test)]
#[test]
fn test_read_param() {
    let input = r#"rel="preconnect";"#.into();
    let (out, rem) = read_param(input);
    assert_eq!(out.name, "rel");
    assert_eq!(out.value.as_deref(), Some("preconnect".into()));
    assert_eq!(rem, ";");
}

/// Implementation of Parsing a Quoted String from Appendinx B.4. in [RFC 8288].
///
/// [RFC 8288]: https://www.rfc-editor.org/rfc/rfc8288.html#appendix-B.4
fn read_quoted_string(input: &BStr) -> (Option<String>, &BStr) {
    // todo(kfj): copy bytes as string only if it contains escaped character

    // B.4.1
    // Let output be an empty string.
    let mut output = String::new();
    // B.4.2
    // If the first character of input is not DQUOTE, return output.
    if input.first().map(|&x| x != b'"').unwrap_or_default() {
        return (None, input);
    }
    // B.4.3
    // Discard the first character.
    let input = &input[1..];
    //
    let mut input = input;
    let input = loop {
        // B.4.4
        // While input has content:
        if input.is_empty() {
            break input;
        }
        input = if input.first().map(|&x| x == b'\\').unwrap_or_default() {
            // B.4.4.1
            // If the first character is a backslash ("\"):
            //
            // B.4.4.1.1
            // Discard the first character.
            let input = &input[1..];
            //
            let input = if input.is_empty() {
                // B.4.4.1.2
                // If there is no more input, return output.
                return (output.into(), input);
            } else {
                // B.4.4.1.3
                // Else, consume the first character and append it to output.
                output.push(input[0] as char);
                &input[1..]
            };
            input
        } else if input.first().map(|&x| x == b'"').unwrap_or_default() {
            // B.4.4.2
            // Else, if the first character is DQUOTE, discard it and return output.
            let input = &input[1..];
            return (output.into(), input);
        } else {
            // B.4.4.3
            // Else, consume the first character and append it to output.
            output.push(input[0] as char);
            &input[1..]
        };
    };
    // B.4.5
    // Return output.
    (output.into(), input)
}

#[cfg(test)]
#[test]
fn test_read_quoted_string() {
    let input = r#""preconnect";"#.into();
    let (out, rem) = read_quoted_string(input);
    assert_eq!(out, Some("preconnect".into()));
    assert_eq!(rem, ";");
}

#[cfg(feature = "http")]
/// This module adds supports for [`http`]'s types.
///
/// [`http`]: https://docs.rs/http
pub mod http {

    use super::*;
    use hyper_http::{header::LINK, HeaderMap, HeaderValue};

    impl<'a> From<&'a HeaderValue> for Links<'a> {
        fn from(s: &'a HeaderValue) -> Self {
            let s = s.as_bytes();
            from_bstr(s)
        }
    }

    /// Create iterator over links from [`HeaderMap`].
    pub fn from_headers(headers: &'_ HeaderMap) -> impl Iterator<Item = Link<'_>> {
        headers
            .get_all(LINK)
            .iter()
            .map(Into::<Links<'_>>::into)
            .flatten()
    }

    #[cfg(test)]
    mod tests {
        use super::*;

        #[test]
        fn test_from_header_value() {
            let value = HeaderValue::from_static(r#"<https://example.com>; rel="preconnect""#);
            let links: Links<'_> = (&value).into();
            let links = links.collect::<Vec<_>>();
            assert_eq!(links.len(), 1);
        }
    }
}

#[cfg(feature = "http-types")]
/// This module adds supports for [`http-types`]'s types.
///
/// [`http-types`]: https://docs.rs/http-types
pub mod http_types {

    use super::*;
    use httrs_http::headers::HeaderValue;

    impl<'a> From<&'a HeaderValue> for Links<'a> {
        fn from(s: &'a HeaderValue) -> Self {
            let s = s.as_str();
            from_bstr(s)
        }
    }

    #[cfg(test)]
    mod tests {
        use super::*;

        #[test]
        fn test_from_header_value() {
            let value =
                HeaderValue::from_bytes(Vec::from(r#"<https://example.com>; rel="preconnect""#))
                    .unwrap();
            let links: Links<'_> = (&value).into();
            let links = links.collect::<Vec<_>>();
            assert_eq!(links.len(), 1);
        }
    }
}
