//! Create async stream for paged response.

// todo(kfj): expand doc

use futures::{ready, Stream, TryFuture};
use pin_project::pin_project;
use std::{
    fmt::Debug,
    pin::Pin,
    task::{Context, Poll},
};

#[derive(PartialEq, Clone, Debug)]
/// Pages cursor.
pub enum Cursor<T> {
    /// Carries cursor for the next page.
    Next(T),

    /// Denotes the end of the pages.
    End,
}

impl<T> From<Option<T>> for Cursor<T> {
    fn from(s: Option<T>) -> Self {
        match s {
            Some(x) => Cursor::Next(x),
            None => Cursor::End,
        }
    }
}

impl<T> From<T> for Cursor<T> {
    fn from(s: T) -> Self {
        Cursor::Next(s)
    }
}

#[derive(PartialEq, Clone, Debug)]
/// Data type for page.
pub struct Page<T, C> {
    /// Data contained in the page.
    pub data: T,

    /// Pages cursor.
    pub cursor: Cursor<C>,
}

/// Type conversion to [`Page`].
pub trait ToPage {
    /// Type for the contained data in the page.
    /// This will be the stream item type.
    type Data;

    /// Type for pages cursor.
    type Cursor;

    /// Convert to [`Page`].
    fn to_page(self) -> Page<Self::Data, Self::Cursor>;
}

impl<T, C> From<(T, Option<C>)> for Page<T, C> {
    fn from((data, cursor): (T, Option<C>)) -> Self {
        let cursor = cursor.into();
        Self { data, cursor }
    }
}

impl<T, C> ToPage for (T, Option<C>) {
    type Data = T;
    type Cursor = C;

    fn to_page(self) -> Page<Self::Data, Self::Cursor> {
        self.into()
    }
}

impl<T, C> ToPage for Page<T, C> {
    type Data = T;
    type Cursor = C;

    fn to_page(self) -> Page<Self::Data, Self::Cursor> {
        self
    }
}

#[pin_project]
#[must_use = "futures do nothing unless you `.await` or poll them"]
/// Stream of pages.
pub struct Pages<F, C, P> {
    factory: F,

    cursor: Option<Cursor<C>>,

    #[pin]
    future: Option<P>,
}

impl<F, C, P> std::fmt::Debug for Pages<F, C, P>
where
    F: Debug,
    C: Debug,
    P: Debug,
{
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        f.debug_struct("Pages")
            .field("factory", &self.factory)
            .field("cursor", &self.cursor)
            .field("future", &self.future)
            .finish()
    }
}

/// Construct new pagination stream.
pub fn with_factory<O, F, C, P>(factory: F) -> Pages<F, C, P>
where
    F: Fn(Option<C>) -> P,
    C: Clone,
    P: TryFuture,
    P::Ok: ToPage<Cursor = C, Data = O>,
{
    let cursor = Default::default();
    let future = Default::default();
    Pages {
        cursor,
        factory,
        future,
    }
}

impl<O, F, C, P, E> Stream for Pages<F, C, P>
where
    F: Fn(Option<C>) -> P,
    C: Clone,
    P: TryFuture<Error = E>,
    P::Ok: ToPage<Cursor = C, Data = O>,
{
    type Item = Result<O, P::Error>;

    fn poll_next(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Option<Self::Item>> {
        let mut this = self.project();
        Poll::Ready(loop {
            if let Some(fut) = this.future.as_mut().as_pin_mut() {
                // producer is not none
                // poll producer
                let res = ready!(fut.try_poll(cx));
                // empty producer, new one will be set in the next loop iteration
                this.future.set(None);
                // propagate error
                let res = match res {
                    Ok(x) => x,
                    Err(x) => {
                        // _terminate_ stream
                        *this.cursor = Cursor::End.into();
                        break Some(Err(x));
                    }
                };
                let Page { cursor, data } = res.to_page();
                // set cursor
                *this.cursor = Some(cursor);
                // _emit_ value
                break Some(Ok(data));
            };
            // producer is none
            // check cursor
            let cur = match this.cursor {
                // end of cursor
                Some(Cursor::End) => {
                    // _emit_ completion
                    break None;
                }
                // not end of cursor
                None => None,
                Some(Cursor::Next(cursor)) => Some(cursor.clone()),
            };
            // create new producer
            let fut = (this.factory)(cur);
            // set producer, this future will be polled in the next loop iteration
            this.future.set(Some(fut));
        })
    }
}

#[cfg(test)]
mod tests {

    use super::*;
    use futures::{StreamExt, TryStream, TryStreamExt};

    fn oks() -> impl TryStream<Ok = usize, Error = &'static str> {
        with_factory(|cur: Option<usize>| async move {
            let idx = cur.unwrap_or_default();
            let paged = match idx {
                0..=1 => Page {
                    cursor: (idx + 1).into(),
                    data: idx + 1,
                },
                2 => Page {
                    cursor: None.into(),
                    data: 3,
                },
                _ => panic!(),
            };
            Result::<_, &str>::Ok(paged)
        })
    }

    fn with_error() -> impl TryStream<Ok = usize, Error = &'static str> {
        with_factory(|cur: Option<usize>| async move {
            let idx = cur.unwrap_or_default();
            match idx {
                0..=1 => {
                    let paged = Page {
                        cursor: (idx + 1).into(),
                        data: idx + 1,
                    };
                    Ok(paged)
                }
                _ => Err("error"),
            }
        })
    }

    #[tokio::test]
    async fn test_try_collect() {
        let pages = oks();

        let items = pages.try_collect().await;

        let items: Vec<_> = items.unwrap();
        assert_eq!(items, vec![1, 2, 3]);
    }

    #[tokio::test]
    async fn test_try_collect_error() {
        let pages = with_error();

        let items = pages.try_collect::<Vec<_>>().await;

        let error = items.unwrap_err();
        assert_eq!(error, "error");
    }

    #[tokio::test]
    async fn test_collect() {
        let pages = oks();

        let items = pages.into_stream().collect::<Vec<Result<_, _>>>().await;

        let items = items.into_iter().collect::<Result<Vec<_>, _>>().unwrap();
        assert_eq!(items, vec![1, 2, 3]);
    }

    #[tokio::test]
    async fn test_collect_error() {
        let pages = with_error();

        let pages = pages.into_stream().collect::<Vec<Result<usize, _>>>().await;

        let (items, errors): (_, Vec<_>) = pages.into_iter().partition(Result::is_ok);
        let items: Vec<_> = items.into_iter().map(Result::unwrap).collect();
        let errors: Vec<_> = errors.into_iter().map(Result::unwrap_err).collect();
        assert_eq!(items, vec![1, 2]);
        assert_eq!(errors, vec!["error"]);
    }
}
