use anyhow::{Error, Result};
use futures::{future, stream, StreamExt, TryStream, TryStreamExt};
use lembaran::{
    stream::pagination,
    web_linking::{Link, Param},
};
use reqwest::{
    header::{ACCEPT, USER_AGENT},
    Url,
};
use serde::Deserialize;

#[derive(Deserialize, Debug)]
struct Repository {
    name: String,
}

fn repositories() -> impl TryStream<Ok = Repository, Error = Error> {
    // Create HTTP client.
    let http = reqwest::Client::new();

    pagination::with_factory(move |url: Option<Url>| {
        let http = http.clone();

        async move {
            // Get URL from previous request or
            let url = match url {
                Some(x) => x,
                None => "https://api.github.com/orgs/rust-lang/repos".parse()?,
            };

            // Create request.
            let req = http
                .get(url)
                .header(
                    USER_AGENT,
                    format!("{}/{}", env!("CARGO_PKG_NAME"), env!("CARGO_PKG_VERSION")),
                )
                .header(ACCEPT, "application/vnd.github.v3+json")
                .send();

            // Send request.
            let res = req.await?.error_for_status()?;

            // Get headers.
            let headers = res.headers();

            // Get links.
            let mut links = lembaran::web_linking::http::from_headers(headers);

            // Find next page URL.
            let next_link = links
                .find(|Link { params, .. }| {
                    params
                        .iter()
                        .find(|Param { name, value }| {
                            *name == "rel" && value.as_deref() == Some("next".into())
                        })
                        .is_some()
                })
                .map(|Link { uri, .. }| uri)
                .map(|x| String::from_utf8_lossy(&**x).parse::<Url>())
                .transpose()?;

            drop(links);

            // Deserialize body.
            let data: Vec<Repository> = res.json().await?;

            // Stream will end when the `next_link` is `None`.
            Ok((data, next_link))
        }
    })
    .flat_map(|x: Result<Vec<_>, _>| match x {
        Ok(x) => stream::iter(x).map(|x| Ok(x)).boxed(),
        Err(x) => stream::once(future::ready(Err(x))).boxed(),
    })
}

#[tokio::main]
async fn main() -> Result<()> {
    repositories()
        .try_for_each(|Repository { name, .. }| {
            // print name to stdout
            println!("{}", name);
            future::ok(())
        })
        .await?;
    Ok(())
}
