use indicatif::ProgressBar;
use jwalk::WalkDir;
use miette::miette;
use miette::Context;
use miette::IntoDiagnostic;
use miette::Result;
use rayon::prelude::*;
use sha2::{Digest, Sha256};
use std::collections::{HashMap, HashSet};
use std::env;
use std::io::BufReader;
use std::io::Write;
use std::path::Path;
use std::path::PathBuf;
use std::{fs, io};
use tracing::warn;
use tracing::{info, Level};
use tracing_subscriber::FmtSubscriber;

fn main() -> Result<()> {
    miette::set_panic_hook();
    let subscriber = FmtSubscriber::builder()
        .with_max_level(Level::TRACE)
        .finish();

    tracing::subscriber::set_global_default(subscriber).expect("setting default subscriber failed");
    let mut args = env::args();
    let directory = args
        .nth(1)
        .ok_or(miette!("please provide directory as a first argument"))?;
    let directory = Path::new(&directory).to_path_buf();
    let database_path = directory.join("legdur.db");

    let all = hash(&directory);
    let db_text = serialise(all)?;

    let old_database_path = backup_old_database_if_needed(&database_path)?;

    write_to_disk(&database_path, db_text)?;

    compare_with_old(old_database_path, database_path)?;
    Ok(())
}

fn compare_with_old(old_database_path: PathBuf, database_path: PathBuf) -> Result<()> {
    if old_database_path.exists() {
        info!(
            "comparing {} with {}",
            &database_path.to_string_lossy(),
            &old_database_path.to_string_lossy()
        );
        let file = fs::File::open(&old_database_path).into_diagnostic()?;
        let reader = BufReader::new(file);
        let old_db: HashMap<String, String> = serde_json::from_reader(reader).into_diagnostic()?;
        let file = fs::File::open(&database_path).into_diagnostic()?;
        let reader = BufReader::new(file);
        let new_db: HashMap<String, String> = serde_json::from_reader(reader).into_diagnostic()?;

        old_db
            .into_par_iter()
            .for_each(|(key, old_value)| match new_db.get(&key) {
                None => warn!("{key} does not exist anymore"),
                Some(new_value) => {
                    if new_value != &old_value {
                        warn!("{key}: {old_value} -> {new_value}")
                    }
                }
            });
    }
    Ok(())
}

fn write_to_disk(database_path: &PathBuf, db_text: String) -> Result<(), miette::Error> {
    let mut db_file = fs::File::create(database_path)
        .into_diagnostic()
        .wrap_err(format!("creating '{}'", database_path.to_string_lossy()))?;
    db_file
        .write_all(db_text.as_bytes())
        .into_diagnostic()
        .wrap_err(format!(
            "writing database to '{}'",
            database_path.to_string_lossy()
        ))?;
    info!("{} saved", database_path.to_string_lossy());
    Ok(())
}

fn serialise(all: HashMap<String, String>) -> Result<String, miette::Error> {
    let db_text = serde_json::to_string(&all)
        .into_diagnostic()
        .wrap_err("serialising")?;
    Ok(db_text)
}

fn hash(directory: &PathBuf) -> HashMap<String, String> {
    info!("scanning '{}'", directory.to_string_lossy());
    let list_of_files = all_files(directory);
    info!("list of files acquired, calculating hashes...");
    let result = calculate_hashes(list_of_files);
    info!("hash calculation complete");
    result
}

fn backup_old_database_if_needed(database_path: &PathBuf) -> Result<PathBuf, miette::Error> {
    let mut old_database_path = database_path.clone();
    old_database_path.set_extension("old");
    if database_path.exists() {
        fs::copy(database_path, &old_database_path)
            .into_diagnostic()
            .wrap_err(format!(
                "copying {} -> {}",
                &database_path.to_string_lossy(),
                &old_database_path.to_string_lossy()
            ))?;
    }
    Ok(old_database_path)
}

fn calculate_hashes(paths: HashSet<PathBuf>) -> HashMap<String, String> {
    let progressbar = ProgressBar::new(paths.len() as u64);

    let result = paths
        .par_iter()
        .filter_map(|path| match fs::File::open(&path) {
            Ok(file) => Some((path, file)),
            Err(_) => None,
        })
        .filter_map(|(path, file)| {
            let mut hasher = Sha256::new();
            let mut file = file;
            if io::copy(&mut file, &mut hasher).is_err() {
                return None;
            }
            let hash = hasher.finalize();
            let hex_text = hex::encode(hash);
            progressbar.inc(1);
            Some((path.to_string_lossy().to_string(), hex_text))
        })
        .collect();

    progressbar.finish_and_clear();

    result
}

fn all_files(directory: &Path) -> HashSet<PathBuf> {
    WalkDir::new(directory)
        .into_iter()
        .filter_map(|entry| entry.ok())
        .filter_map(|entry| match entry.file_type().is_dir() {
            false => Some(entry.path().to_path_buf()),
            true => None,
        })
        .collect()
}
