use rust_decimal::prelude::MathematicalOps;
use rust_decimal::prelude::RoundingStrategy;
use rust_decimal::Decimal;
use rust_decimal_macros::dec;

pub fn calculate_interest_daily(principal_balance: Decimal, apr: Decimal, number_of_days: u16) -> Decimal {
	// We don't pay out interest on negative balances
	if(principal_balance <= dec!(0)) {
		return dec!(0);
	}

	// multiply apr by 0.01 because it's stored as a percentage
	let dpr = apr * dec!(0.01) / dec!(365.25);
	let interest_base = Decimal::ONE + dpr;
	let interest_pct = interest_base.powu(number_of_days as u64) - Decimal::ONE;
	(principal_balance * interest_pct).round_dp_with_strategy(2, RoundingStrategy::MidpointAwayFromZero)
}

#[allow(dead_code)]
pub fn calculate_interest_monthly(principal_balance: Decimal, apr: Decimal) -> Decimal {
	// We don't pay out interest on negative balances
	if(principal_balance <= dec!(0)) {
		return dec!(0);
	}

	// multiply apr by 0.01 because it's stored as a percentage
	let mpr = apr * dec!(0.01) / dec!(12);
	(principal_balance * mpr).round_dp_with_strategy(2, RoundingStrategy::MidpointAwayFromZero)
}

#[cfg(test)]
mod tests {
	use rust_decimal::prelude::*;
	use super::*;

	fn interest(value: f64, apr: f32, number_of_days: u16) -> f64 {
		calculate_interest_daily(Decimal::from_f64(value).unwrap(), Decimal::from_f32(apr).unwrap(), number_of_days).to_f64().unwrap()
	}

	fn interest_monthly(value: f64, apr: f32) -> f64 {
		calculate_interest_monthly(Decimal::from_f64(value).unwrap(), Decimal::from_f32(apr).unwrap()).to_f64().unwrap()
	}

	#[test]
	fn test_calculate_interest_daily() {
		assert_eq!(8.22, interest(10000.0, 1.0, 30));
		assert_eq!(7.94, interest(9659.18, 1.0, 30));
	}

	#[test]
	fn test_calculate_interest_monthly() {
		assert_eq!(335.28, interest_monthly(89407.33, 4.5));
		assert_eq!(334.11, interest_monthly(89094.75, 4.5));
	}
}


