use actix_web::web::Data;
use actix_web::web::Json;
use actix_web::web::Path;
use itertools::Itertools;
use rust_decimal::Decimal;
use rust_decimal_macros::dec;
use serde::Deserialize;
use serde::Serialize;
use serde_repr::Deserialize_repr;
use sqlx::FromRow;
use sqlx::MySqlPool;
use time::OffsetDateTime;

use crate::account::Account;
use crate::util::calculate_interest_daily;
use crate::Error;
use crate::Tx;

#[derive(Clone, Debug, Deserialize)]
pub struct CreateTransaction {
	kind: TransactionType,
	amount: Decimal
}

#[derive(Clone, Copy, Debug, Deserialize_repr)]
#[repr(u8)]
pub enum TransactionType {
	Payment = 1,
	Debit = 2
}

#[derive(Clone, Debug, Eq, FromRow, Serialize)]
pub struct Transaction {
	account_id: u32,
	#[serde(with = "time::serde::timestamp")]
	timestamp: OffsetDateTime,
	principal: Decimal,
	accrued_interest: Decimal,
	paid_interest: Decimal
}

impl PartialEq for Transaction {
	fn eq(&self, other: &Self) -> bool {
		// Primarily for testing; just compare the dates and amounts.
		// This could plausibly fail if they're kicked off a couple microseconds before midnight UTC.
		self.timestamp.date() == other.timestamp.date() &&
			self.principal == other.principal &&
			self.accrued_interest == other.accrued_interest &&
			self.paid_interest == other.paid_interest
	}
}

impl PartialOrd for Transaction {
	fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
		self.timestamp.partial_cmp(&other.timestamp)
	}
}

impl Ord for Transaction {
	fn cmp(&self, other: &Self) -> std::cmp::Ordering {
		self.timestamp.cmp(&other.timestamp)
	}
}

pub async fn create(pool: Data<MySqlPool>, account_id: Path<u32>, transaction: Json<CreateTransaction>) -> Result<Json<Transaction>, Error> {
	let mut tx = pool.begin().await?;
	let transaction = Transaction::create(&mut tx, *account_id, transaction.kind, transaction.amount).await?;
	tx.commit().await?;
	Ok(Json(transaction))
}

pub async fn get_all(pool: Data<MySqlPool>, account_id: Path<u32>) -> Result<Json<Vec<Transaction>>, Error> {
	let mut tx = pool.begin().await?;
	Ok(Json(Transaction::get_by_account(&mut tx, *account_id).await?))
}

impl Transaction {
	// TODO:  Allow transactions at times other than "right now"
	pub async fn create(tx: &mut Tx, account_id: u32, kind: TransactionType, amount: Decimal) -> Result<Self, Error> {
		let now = OffsetDateTime::now_utc();
		let mut account = Account::get(&mut *tx, account_id).await?;
		let previous_transaction_date = account.transactions.get(0).unwrap().timestamp;
		let this = match kind {
			TransactionType::Payment => Self::calculate_payment(account.principal_balance, account.interest_balance, account.apr, amount, previous_transaction_date, now),
			TransactionType::Debit => Self::calculate_debit(account.principal_balance, account.apr, amount, previous_transaction_date, now)
		};
		sqlx::query!("INSERT INTO accounts_transactions VALUES (?, ?, ?, ?, ?)", this.timestamp, account.id, this.principal, this.accrued_interest, this.paid_interest).execute(&mut *tx).await?;
		if(this.timestamp < account.transactions.iter().last().unwrap().timestamp) {
			let this_timestamp = this.timestamp;
			account.transactions.push(this);
			account.transactions.sort_unstable();
			let (_, first_changed_idx) = reconcile_history(&mut account.transactions, account.apr);
			if let Some(idx) = first_changed_idx {
				for txn in account.transactions[idx..].iter() {
					sqlx::query!("UPDATE accounts_transactions SET principal = ?, accrued_interest = ?, paid_interest = ? WHERE account_id = ? AND timestamp = ?", txn.principal, txn.accrued_interest, txn.paid_interest, account.id, txn.timestamp).execute(&mut *tx).await?;
				}
			}
			return Ok(account.transactions.into_iter().find(|txn| txn.timestamp == this_timestamp).unwrap());
		}
		Ok(this)
	}

	pub async fn get_by_account(tx: &mut Tx, account_id: u32) -> Result<Vec<Transaction>, Error> {
		Ok(sqlx::query_as!(Self, "SELECT * FROM accounts_transactions WHERE account_id = ? ORDER BY `timestamp` DESC", account_id).fetch_all(tx).await?)
	}

	fn calculate_debit(principal_balance: Decimal, apr: Decimal, amount: Decimal, previous_transaction_date: OffsetDateTime, this_transaction_date: OffsetDateTime) -> Self /* {{{ */ {
		let interval = (this_transaction_date - previous_transaction_date).whole_days() as u16;
		Self{
			account_id: 0,
			timestamp: this_transaction_date,
			principal: amount,
			accrued_interest: calculate_interest_daily(principal_balance, apr, interval),
			paid_interest: dec!(0)
		}
	} // }}}

	fn calculate_payment(principal_balance: Decimal, interest_balance: Decimal, apr: Decimal, amount: Decimal, previous_transaction_date: OffsetDateTime, this_transaction_date: OffsetDateTime) -> Self /* {{{ */ {
		let interval = (this_transaction_date - previous_transaction_date).whole_days() as u16;
		let paid_interest = std::cmp::min(interest_balance, amount);
		let principal = amount - paid_interest;
		Self{
			account_id: 0,
			timestamp: this_transaction_date,
			principal: -principal,
			accrued_interest: calculate_interest_daily(principal_balance, apr, interval),
			paid_interest
		}
	} // }}}
}

/// INVARIANT:  transactions must be sorted by date, oldest first
///
/// Returns the ending principal balance and the index of the first changed transaction.
fn reconcile_history(transactions: &mut [Transaction], apr: Decimal) -> (Decimal, Option<usize>) {
	if(transactions.is_empty()) {
		return (dec!(0), None);
	}
	transactions[0].accrued_interest = dec!(0);
	transactions[0].paid_interest = dec!(0);
	let mut first_changed_idx = None;
	let mut principal_balance = dec!(0);
	let mut rolling_interest = dec!(0);
	for (i, j) in (0..transactions.len()).tuple_windows() {
		let previous_tx = transactions[i].clone();
		let mut tx = &mut transactions[j];
		principal_balance += previous_tx.principal;
		let interval = (tx.timestamp - previous_tx.timestamp).whole_days() as u16;
		let interest = calculate_interest_daily(principal_balance, apr, interval);
		rolling_interest += interest;
		let paid_interest;
		let principal;
		if(tx.principal <= dec!(0)) {
			let amount = -tx.principal + tx.paid_interest;
			paid_interest = std::cmp::min(rolling_interest, amount);
			rolling_interest -= paid_interest;
			principal = -(amount - paid_interest);
		} else {
			paid_interest = dec!(0);
			principal = tx.principal;
		}
		if(interest != tx.accrued_interest || paid_interest != tx.paid_interest || principal != tx.principal) {
			tx.accrued_interest = interest;
			tx.paid_interest = paid_interest;
			tx.principal = principal;
			if(first_changed_idx.is_none()) {
				first_changed_idx = Some(i + 1);
			}
		}
	}
	principal_balance += transactions.iter().last().unwrap().principal;
	(principal_balance, first_changed_idx)
}

#[cfg(test)]
mod tests {
	use std::time::Duration;
	use rand::seq::SliceRandom;
	use rust_decimal::prelude::*;
	use rust_decimal::Decimal;
	use time::OffsetDateTime;
	use super::*;

	fn txn(age_days: u64, principal: f64, accrued_interest: f64, paid_interest: f64) -> Transaction {
		Transaction{
			account_id: 0,
			timestamp: OffsetDateTime::now_utc() - Duration::from_secs(age_days * 86400),
			principal: Decimal::from_f64(principal).unwrap(),
			accrued_interest: Decimal::from_f64(accrued_interest).unwrap(),
			paid_interest: Decimal::from_f64(paid_interest).unwrap()
		}
	}

	#[test]
	fn reconcile_append_transaction_no_change() /* {{{ */ {
		let mut transactions = vec![
			txn(60, 10000.0, 0.0, 0.0),
			txn(30, -34.99, 49.4, 49.4)
		];
		transactions.push(txn(0, -35.16, 49.23, 49.23));
		let old_transactions = transactions.clone();
		let (principal, first_changed_idx) = reconcile_history(&mut transactions, dec!(6));
		assert_eq!(principal, dec!(9929.85));
		assert_eq!(first_changed_idx, None);
		assert_eq!(transactions[0], txn(60, 10000.0, 0.0, 0.0));
		assert_eq!(transactions[1], txn(30, -34.99, 49.4, 49.4));
		assert_eq!(transactions[2], txn(0, -35.16, 49.23, 49.23));
		assert_eq!(old_transactions, transactions);
	} // }}}

	#[test]
	fn reconcile_append_transaction_naive() /* {{{ */ {
		let mut transactions = vec![
			txn(90, 1000.0, 0.0, 0.0),
			txn(60, -79.67, 8.25, 8.25),
			txn(30, -80.33, 7.59, 7.59)
		];
		transactions.push(txn(0, -87.92, 0.0, 0.0));
		let (principal, first_changed_idx) = reconcile_history(&mut transactions, dec!(10));
		assert_eq!(principal, dec!(759.01));
		assert_eq!(first_changed_idx, Some(3));
		assert_eq!(transactions[0], txn(90, 1000.0, 0.0, 0.0));
		assert_eq!(transactions[1], txn(60, -79.67, 8.25, 8.25));
		assert_eq!(transactions[2], txn(30, -80.33, 7.59, 7.59));
		assert_eq!(transactions[3], txn(0, -80.99, 6.93, 6.93));
	} // }}}

	#[test]
	fn reconcile_fix_transactions() /* {{{ */ {
		let mut transactions = vec![
			txn(180, 1337.0, 0.0, 0.0),
			txn(150, -58.06, 0.0, 0.0),
			txn(120, -58.06, 0.0, 0.0),
			txn(90, -58.06, 0.0, 0.0),
			txn(60, -58.06, 0.0, 0.0),
			txn(30, -58.06, 0.0, 0.0),
			txn(0, -58.06, 0.0, 0.0)
		];
		let (principal, first_changed_idx) = reconcile_history(&mut transactions, dec!(4));
		assert_eq!(principal, dec!(1012.38));
		assert_eq!(first_changed_idx, Some(1));
		assert_eq!(transactions[0], txn(180, 1337.0, 0.0, 0.0));
		assert_eq!(transactions[1], txn(150, -53.66, 4.40, 4.40));
		assert_eq!(transactions[2], txn(120, -53.84, 4.22, 4.22));
		assert_eq!(transactions[3], txn(90, -54.01, 4.05, 4.05));
		assert_eq!(transactions[4], txn(60, -54.19, 3.87, 3.87));
		assert_eq!(transactions[5], txn(30, -54.37, 3.69, 3.69));
		assert_eq!(transactions[6], txn(0, -54.55, 3.51, 3.51));
	} // }}}

	#[test]
	fn reconcile_fix_transactions_revolving() /* {{{ */ {
		let mut transactions = vec![
			txn(180, 50.0, 0.0, 0.0),
			txn(178, 25.0, 0.0, 0.0),
			txn(177, 32.0, 0.0, 0.0),
			txn(170, 7.99, 0.0, 0.0),
			txn(169, 6.5, 0.0, 0.0),
			txn(160, 159.95, 0.0, 0.0),
			txn(150, -100.0, 0.0, 0.0),
			txn(122, -1.0, 0.0, 0.0),
			txn(120, -100.0, 0.0, 0.0),
			txn(115, 500.0, 0.0, 0.0),
			txn(111, 27.84, 0.0, 0.0),
			txn(100, -50.0, 0.0, 0.0),
			txn(90, -600.0, 0.0, 0.0),
			txn(60, 1.5, 0.0, 0.0),
			txn(30, 2.0, 0.0, 0.0),
			txn(29, 30.0, 0.0, 0.0),
			txn(25, 30.0, 0.0, 0.0),
			txn(0, -31.19, 0.0, 0.0)
		];
		let (principal, first_changed_idx) = reconcile_history(&mut transactions, dec![13]);
		assert_eq!(principal, dec!(0));
		assert_eq!(first_changed_idx, Some(1));
		assert_eq!(transactions[0], txn(180, 50.0, 0.0, 0.0));
		assert_eq!(transactions[1], txn(178, 25.0, 0.04, 0.0));
		assert_eq!(transactions[2], txn(177, 32.0, 0.03, 0.0));
		assert_eq!(transactions[3], txn(170, 7.99, 0.27, 0.0));
		assert_eq!(transactions[4], txn(169, 6.5, 0.04, 0.0));
		assert_eq!(transactions[5], txn(160, 159.95, 0.39, 0.0));
		assert_eq!(transactions[6], txn(150, -98.23, 1.0, 1.77));
		assert_eq!(transactions[7], txn(122, 0.0, 1.83, 1.0));
		assert_eq!(transactions[8], txn(120, -99.04, 0.13, 0.96));
		assert_eq!(transactions[9], txn(115, 500.0, 0.15, 0.0));
		assert_eq!(transactions[10], txn(111, 27.84, 0.83, 0.0));
		assert_eq!(transactions[11], txn(100, -46.62, 2.4, 3.38));
		assert_eq!(transactions[12], txn(90, -597.98, 2.02, 2.02));
		assert_eq!(transactions[13], txn(60, 1.5, 0.0, 0.0));
		assert_eq!(transactions[14], txn(30, 2.0, 0.0, 0.0));
		assert_eq!(transactions[15], txn(29, 30.0, 0.0, 0.0));
		assert_eq!(transactions[16], txn(25, 30.0, 0.0, 0.0));
		assert_eq!(transactions[17], txn(0, -30.91, 0.28, 0.28));
	} // }}}

	#[test]
	fn reconcile_fix_transactions_revolving_reorder() /* {{{ */ {
		let mut transactions = vec![
			txn(180, 50.0, 0.0, 0.0),
			txn(178, 25.0, 0.0, 0.0),
			txn(177, 32.0, 0.0, 0.0),
			txn(170, 7.99, 0.0, 0.0),
			txn(169, 6.5, 0.0, 0.0),
			txn(160, 159.95, 0.0, 0.0),
			txn(150, -100.0, 0.0, 0.0),
			txn(122, -1.0, 0.0, 0.0),
			txn(120, -100.0, 0.0, 0.0),
			txn(115, 500.0, 0.0, 0.0),
			txn(111, 27.84, 0.0, 0.0),
			txn(100, -50.0, 0.0, 0.0),
			txn(90, -600.0, 0.0, 0.0),
			txn(60, 1.5, 0.0, 0.0),
			txn(30, 2.0, 0.0, 0.0),
			txn(29, 30.0, 0.0, 0.0),
			txn(25, 30.0, 0.0, 0.0),
			txn(0, -31.19, 0.0, 0.0)
		];
		let mut rng = rand::thread_rng();
		transactions.shuffle(&mut rng);
		reconcile_history(&mut transactions, dec![13]);
		transactions.sort_unstable();
		let (principal, first_changed_idx) = reconcile_history(&mut transactions, dec![13]);
		assert_eq!(principal, dec!(0));
		assert_eq!(first_changed_idx, Some(1));
		assert_eq!(transactions[0], txn(180, 50.0, 0.0, 0.0));
		assert_eq!(transactions[1], txn(178, 25.0, 0.04, 0.0));
		assert_eq!(transactions[2], txn(177, 32.0, 0.03, 0.0));
		assert_eq!(transactions[3], txn(170, 7.99, 0.27, 0.0));
		assert_eq!(transactions[4], txn(169, 6.5, 0.04, 0.0));
		assert_eq!(transactions[5], txn(160, 159.95, 0.39, 0.0));
		assert_eq!(transactions[6], txn(150, -98.23, 1.0, 1.77));
		assert_eq!(transactions[7], txn(122, 0.0, 1.83, 1.0));
		assert_eq!(transactions[8], txn(120, -99.04, 0.13, 0.96));
		assert_eq!(transactions[9], txn(115, 500.0, 0.15, 0.0));
		assert_eq!(transactions[10], txn(111, 27.84, 0.83, 0.0));
		assert_eq!(transactions[11], txn(100, -46.62, 2.4, 3.38));
		assert_eq!(transactions[12], txn(90, -597.98, 2.02, 2.02));
		assert_eq!(transactions[13], txn(60, 1.5, 0.0, 0.0));
		assert_eq!(transactions[14], txn(30, 2.0, 0.0, 0.0));
		assert_eq!(transactions[15], txn(29, 30.0, 0.0, 0.0));
		assert_eq!(transactions[16], txn(25, 30.0, 0.0, 0.0));
		assert_eq!(transactions[17], txn(0, -30.91, 0.28, 0.28));
	} // }}}
}

