#![allow(unused_parens)]
use actix_web::middleware;
use actix_web::web;
use actix_web::App;
use actix_web::HttpServer;
use clap::Parser;

mod error;
use error::Error;
mod account;
mod transaction;
mod util;
mod assets;

type Tx = sqlx::Transaction<'static, sqlx::MySql>;

#[derive(Debug, Parser)]
struct Config {
	#[clap(long, env = "DATABASE_URL")]
	database_url: String,
	#[clap(long, env = "LEDGRR_PORT", default_value_t = 80)]
	port: u16
}

#[actix_web::main]
async fn main() {
	let config = Config::parse();
	tracing_subscriber::fmt()
		.with_env_filter(tracing_subscriber::EnvFilter::from_default_env())
		.compact()
		.init();
	let pool = sqlx::MySqlPool::connect(&config.database_url).await.unwrap();
	sqlx::migrate!("./schema").run(&pool).await.unwrap();
	let pool = web::Data::new(pool);
	let server = HttpServer::new(move || App::new()
		.app_data(pool.clone())
		.wrap(middleware::Logger::default())
		.wrap(actix_cors::Cors::permissive())
		.route("/", web::get().to(assets::index))
		.service(web::resource("/{_:.*\\..*}").route(web::get().to(assets::static_file)))
		.service(web::scope("/api")
			.route("/accounts", web::post().to(account::create))
			.route("/accounts", web::get().to(account::get_all))
			.route("/accounts/{id}", web::get().to(account::get_by_id))
			.route("/accounts/{id}/transactions", web::post().to(transaction::create))
			.route("/accounts/{id}/transactions", web::get().to(transaction::get_all))
		)
	);
	server
		.shutdown_timeout(10)
		.bind(&format!("0.0.0.0:{}", config.port))
		.unwrap()
		.run()
		.await
		.unwrap();
}

