use actix_web::web::Data;
use actix_web::web::Json;
use actix_web::web::Path;
use rust_decimal::Decimal;
use serde::Deserialize;
use serde::Serialize;
use sqlx::FromRow;
use sqlx::MySqlPool;
use time::OffsetDateTime;

use crate::transaction::Transaction;
use crate::util::calculate_interest_daily;
use crate::Error;
use crate::Tx;

#[derive(Debug, Deserialize)]
pub struct CreateAccount {
	name: String,
	initial_balance: Decimal,
	apr: Decimal
}

#[derive(Debug, Serialize)]
pub struct Account {
	pub id: u32,
	pub name: String,
	pub apr: Decimal,
	pub principal_balance: Decimal,
	pub interest_balance: Decimal,
	pub transactions: Vec<Transaction>
}

pub async fn create(pool: Data<MySqlPool>, account: Json<CreateAccount>) -> Result<Json<Account>, Error> {
	let mut tx = pool.begin().await?;
	let account = Account::create(&mut tx, account.into_inner()).await?;
	tx.commit().await?;
	Ok(Json(account))
}

pub async fn get_all(pool: Data<MySqlPool>) -> Result<Json<Vec<Account>>, Error> {
	let mut tx = pool.begin().await?;
	Ok(Json(Account::get_all(&mut tx).await?))
}

pub async fn get_by_id(pool: Data<MySqlPool>, id: Path<u32>) -> Result<Json<Account>, Error> {
	let mut tx = pool.begin().await?;
	Ok(Json(Account::get(&mut tx, *id).await?))
}

#[derive(Clone, Debug, FromRow)]
pub struct AccountRow {
	id: u32,
	name: String,
	apr: Decimal,
	principal_balance: Option<Decimal>,
	interest_balance: Option<Decimal>,
	most_recent_transaction: OffsetDateTime
}

impl Account {
	pub async fn create(tx: &mut Tx, account: CreateAccount) -> Result<Self, Error> {
		let result = sqlx::query!("INSERT INTO accounts VALUES (DEFAULT, ?, ?, ?)", account.name, account.initial_balance, account.apr).execute(&mut *tx).await?;
		let id = result.last_insert_id() as u32;
		sqlx::query!("INSERT INTO accounts_transactions VALUES (NOW(), ?, ?, 0, 0)", id, account.initial_balance).execute(&mut *tx).await?;
		Self::get(tx, id).await
	}

	pub async fn get_all(tx: &mut Tx) -> Result<Vec<Self>, Error> /* {{{ */ {
		let rows = sqlx::query_as!(AccountRow, "
			SELECT
				accounts.id,
				accounts.name,
				accounts.apr,
				txn_agg.principal AS principal_balance,
				txn_agg.interest AS interest_balance,
				txn_recent.timestamp AS most_recent_transaction
			FROM
				accounts
				INNER JOIN (
					SELECT account_id, MAX(`timestamp`) AS most_recent_timestamp, SUM(principal) AS principal, SUM(accrued_interest - paid_interest) AS interest FROM accounts_transactions GROUP BY account_id
				) AS txn_agg ON accounts.id = txn_agg.account_id
				INNER JOIN accounts_transactions AS txn_recent ON txn_recent.account_id = accounts.id AND txn_recent.timestamp = txn_agg.most_recent_timestamp
		").fetch_all(&mut *tx).await?;
		let mut accounts = Vec::with_capacity(rows.len());
		for row in rows {
			// TODO:  Get all transactions in one query and iterate through them in here to assign to accounts
			accounts.push(Self::from_row(&mut *tx, row).await?);
		}
		Ok(accounts)
	} // }}}

	pub async fn get(tx: &mut Tx, id: u32) -> Result<Self, Error> /* {{{ */ {
		let row = sqlx::query_as!(AccountRow, "
			SELECT
				accounts.id,
				accounts.name,
				accounts.apr,
				txn_agg.principal AS principal_balance,
				txn_agg.interest AS interest_balance,
				txn_recent.timestamp AS most_recent_transaction
			FROM
				accounts
				INNER JOIN (
					SELECT account_id, MAX(`timestamp`) AS most_recent_timestamp, SUM(principal) AS principal, SUM(accrued_interest - paid_interest) AS interest FROM accounts_transactions GROUP BY account_id
				) AS txn_agg ON accounts.id = txn_agg.account_id
				INNER JOIN accounts_transactions AS txn_recent ON txn_recent.account_id = accounts.id AND txn_recent.timestamp = txn_agg.most_recent_timestamp
			WHERE
				accounts.id = ?
		", id).fetch_optional(&mut *tx).await?.ok_or(Error::AccountNotFound(id))?;
		Self::from_row(tx, row).await
	} // }}}

	async fn from_row(tx: &mut Tx, row: AccountRow) -> Result<Self, Error> {
		let days_since_last_transaction = (OffsetDateTime::now_utc() - row.most_recent_transaction).whole_days() as u16;
		let principal_balance = row.principal_balance.unwrap_or_default();
		Ok(Self{
			id: row.id,
			name: row.name,
			apr: row.apr,
			principal_balance,
			interest_balance: row.interest_balance.unwrap_or_default() + calculate_interest_daily(principal_balance, row.apr, days_since_last_transaction),
			transactions: Transaction::get_by_account(tx, row.id).await?
		})
	}
}

