use crate::account_balance::AccountBalance;
use crate::balance::Balance;
use crate::simplified_ledger::SimplificationError;
use ledger_parser::{
    Amount, Commodity, CommodityPosition, Posting, PostingAmount, Reality, Transaction,
};
use ledger_parser::{Balance::Amount as BalanceAmount, Balance::Zero as BalanceZero};
use rust_decimal::Decimal;

/// Fails if any transactions are unbalanced, any balance assertions fail, or if an unbalanced
/// virtual posting (account name in `()`) has no amount.
///
/// "Balance assertions" are postings with both amount and balance provided. The calculated
/// amount using the balance must match the given amount.
pub fn calculate_amounts_from_balances(
    transactions: &mut Vec<Transaction>,
) -> Result<(), SimplificationError> {
    let mut running_balance = Some(Balance::new());

    for transaction in transactions {
        calculate_transaction_amounts(transaction, &mut running_balance)?;
    }

    Ok(())
}

/// Fails if any transactions are unbalanced, or if an unbalanced virtual posting
/// (account name in `()`) has no amount.
///
/// Ignores `balance`s. Fails if they are necessary to fill in any omitted `amount`s.
pub fn calculate_omitted_amounts(transaction: &mut Transaction) -> Result<(), SimplificationError> {
    calculate_transaction_amounts(transaction, &mut None)
}

fn calculate_transaction_amounts(
    transaction: &mut Transaction,
    running_balance: &mut Option<Balance>,
) -> Result<(), SimplificationError> {
    let original_transaction = transaction.clone();

    let mut real_transaction_balance = AccountBalance::new();
    let mut virtual_transaction_balance = AccountBalance::new();

    // For postings with amounts, add those amounts to the transaction and running balances.
    for posting in &transaction.postings {
        if let Some(PostingAmount { ref amount, .. }) = posting.amount {
            match posting.reality {
                Reality::Real => real_transaction_balance += amount,
                Reality::BalancedVirtual => virtual_transaction_balance += amount,
                Reality::UnbalancedVirtual => (),
            }

            if let Some(running_balance) = running_balance {
                running_balance.add_amount(&posting.account, amount);
            }
        }
    }

    if let Some(running_balance) = running_balance {
        // For postings with balances, calculate their amounts from the running balance.
        for posting in &mut transaction.postings {
            if posting.balance.is_some() {
                calculate_posting_amounts_from_balances(
                    &original_transaction,
                    posting,
                    running_balance,
                    &mut real_transaction_balance,
                    &mut virtual_transaction_balance,
                )?;
            }
        }
    }

    // Discard any postings with empty commodity name generated by the previous step.
    transaction.postings.retain(|posting| {
        posting
            .amount
            .as_ref()
            .map_or(true, |amt| !amt.amount.commodity.name.is_empty())
    });

    // For any postings still without amounts, calculate the amount from the transaction balance.
    // This may generate multiple new postings if there are multiple commodities with unbalanced
    // amounts remaining.
    let mut new_postings = Vec::new();
    for posting in &transaction.postings {
        if posting.amount.is_none() {
            let balancing_postings = calculate_omitted_amounts_for_posting(
                &original_transaction,
                posting,
                &mut real_transaction_balance,
                &mut virtual_transaction_balance,
            )?;

            if let Some(running_balance) = running_balance {
                for balancing_posting in &balancing_postings {
                    running_balance.add_amount(
                        &balancing_posting.account,
                        &balancing_posting
                            .amount
                            .as_ref()
                            .expect("we just calculated all the amounts")
                            .amount,
                    );
                }
            }

            new_postings.extend(balancing_postings);
        } else {
            new_postings.push(posting.clone());
        }
    }

    // Check that all real and balanced virtual postings now balance.
    if !real_transaction_balance.is_zero() || !virtual_transaction_balance.is_zero() {
        // TODO: Handle the case where there are exactly two currencies that are non-zero, by
        // creating a commodity price for this date that makes the transaction balance.
        return Err(SimplificationError::UnbalancedTransaction(
            transaction.clone(),
        ));
    }

    transaction.postings = new_postings;

    Ok(())
}

fn calculate_posting_amounts_from_balances(
    transaction: &Transaction,
    posting: &mut Posting,
    running_balance: &mut Balance,
    real_transaction_balance: &mut AccountBalance,
    virtual_transaction_balance: &mut AccountBalance,
) -> Result<(), SimplificationError> {
    if let Some(posting_balance) = &posting.balance {
        let account_balance = running_balance.account_balances.get(&posting.account);

        let (commodity, balance_target, current_balance) = match posting_balance {
            BalanceAmount(balance_amount) => {
                // Balance given, find account balance with matching commodity.
                match account_balance.and_then(|b| b.amounts.get(&balance_amount.commodity.name)) {
                    Some(current_balance) => {
                        // Balance provided and account has balance.
                        (
                            balance_amount.commodity.clone(),
                            balance_amount.quantity,
                            current_balance.quantity,
                        )
                    }
                    None => {
                        // Balance provided but account has no balance, assume zero.
                        (
                            balance_amount.commodity.clone(),
                            balance_amount.quantity,
                            Decimal::ZERO,
                        )
                    }
                }
            }
            BalanceZero => match account_balance.map_or(0, |b| b.amounts.len()) {
                0 => {
                    // Zero balance given and account is empty (no commodities). Discard this
                    // posting by using an empty commodity which we will filter out later.
                    (
                        Commodity {
                            name: "".to_owned(),
                            position: CommodityPosition::Left,
                        },
                        Decimal::ZERO,
                        Decimal::ZERO,
                    )
                }
                1 => {
                    // Zero balance given and account has exactly one commodity.
                    let current_balance = account_balance
                        .expect("just checked there's an account")
                        .amounts
                        .values()
                        .next()
                        .expect("just checked there's one commodity");
                    (
                        current_balance.commodity.clone(),
                        Decimal::ZERO,
                        current_balance.quantity,
                    )
                }
                _ => {
                    // Zero balance given and account has multiple commodities. If a balance assertion
                    // (amount also provided), it would fail due to non-zero account balance. If not a
                    // balance assertion, Ledger-cli doesn't handle this case. We could by creating
                    // multiple postings, but since Ledger-cli doesn't we haven't.
                    return Err(SimplificationError::ZeroBalanceMultipleCurrencies(
                        transaction.clone(),
                    ));
                }
            },
        };

        if posting.amount.is_some() {
            // Posting has an amount. It will already have been included in current_balance
            // so just check that the posting balance is equal to that.
            if let BalanceAmount(posting_balance) = posting_balance {
                if posting_balance.quantity != current_balance {
                    return Err(SimplificationError::BalanceAssertionFailed(
                        transaction.clone(),
                    ));
                }
            } else if current_balance != Decimal::ZERO {
                return Err(SimplificationError::ZeroBalanceAssertionFailed(
                    transaction.clone(),
                ));
            }
        } else {
            // Posting has no amount, but has a balance, use the amount calculated from the
            // balance.
            let amount = Amount {
                commodity,
                quantity: balance_target - current_balance,
            };

            match posting.reality {
                Reality::Real => *real_transaction_balance += &amount,
                Reality::BalancedVirtual => *virtual_transaction_balance += &amount,
                Reality::UnbalancedVirtual => (),
            }

            running_balance.add_amount(&posting.account, &amount);

            posting.amount = Some(PostingAmount {
                amount,
                lot_price: None,
                price: None,
            });
        }

        posting.balance = None;
    }

    Ok(())
}

fn calculate_omitted_amounts_for_posting(
    transaction: &Transaction,
    posting: &Posting,
    real_transaction_balance: &mut AccountBalance,
    virtual_transaction_balance: &mut AccountBalance,
) -> Result<Vec<Posting>, SimplificationError> {
    // Posting has no amount or balance, figure out what value (or values, if multi-currency)
    // to insert to make the transaction balance
    let transaction_balance = match posting.reality {
        Reality::Real => real_transaction_balance,
        Reality::BalancedVirtual => virtual_transaction_balance,
        Reality::UnbalancedVirtual => {
            return Err(SimplificationError::UnbalancedVirtualWithNoAmount(
                transaction.clone(),
            ))
        }
    };

    // Make sure the ordering of generated postings is stable, mostly for the tests
    let mut sorted_amounts: Vec<_> = transaction_balance.amounts.values().collect();
    sorted_amounts.sort_by_key(|amount| &amount.commodity.name);

    let new_postings = sorted_amounts
        .iter()
        .map(|amount| {
            let amount = Amount {
                commodity: amount.commodity.clone(),
                quantity: -amount.quantity,
            };

            Posting {
                account: posting.account.clone(),
                reality: posting.reality,
                status: posting.status,
                comment: posting.comment.clone(),
                amount: Some(PostingAmount {
                    amount,
                    lot_price: None,
                    price: None,
                }),
                balance: None,
            }
        })
        .collect();

    // Transaction balance is zero by definition now
    *transaction_balance = AccountBalance::new();

    Ok(new_postings)
}

#[cfg(test)]
mod tests {
    use super::*;
    use ledger_parser::LedgerItem;

    fn parse_transaction(input: &str) -> Transaction {
        ledger_parser::parse(input)
            .unwrap()
            .items
            .into_iter()
            .find_map(|item| match item {
                LedgerItem::Transaction(txn) => Some(txn),
                _ => None,
            })
            .unwrap()
    }

    fn parse_transactions(input: &str) -> Vec<Transaction> {
        ledger_parser::parse(input)
            .unwrap()
            .items
            .into_iter()
            .filter_map(|item| match item {
                LedgerItem::Transaction(txn) => Some(txn),
                _ => None,
            })
            .collect()
    }

    #[test]
    fn test_calculate_omitted_amounts_no_changes() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC   $1.20
  TEST:DEF  $-1.20
"#,
        );
        let original_transaction = transaction.clone();
        assert_eq!(calculate_omitted_amounts(&mut transaction), Ok(()));
        assert_eq!(transaction, original_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_single_commodity() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC  $1.20
  TEST:DEF        ; Comment
"#,
        );
        let expected_transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC   $1.20
  TEST:DEF  $-1.20 ; Comment
"#,
        );
        assert_eq!(calculate_omitted_amounts(&mut transaction), Ok(()));
        assert_eq!(transaction, expected_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_multi_commodity() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC  $1.20
  TEST:DEF  £3.40
  TEST:GHI        ; Comment
"#,
        );
        let expected_transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC   $1.20
  TEST:DEF   £3.40
  TEST:GHI  $-1.20 ; Comment
  TEST:GHI  £-3.40 ; Comment
"#,
        );
        assert_eq!(calculate_omitted_amounts(&mut transaction), Ok(()));
        assert_eq!(transaction, expected_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_error_unbalanced_transaction() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC   $1.20
  TEST:DEF  $-1.21
"#,
        );
        let original_transaction = transaction.clone();
        assert_eq!(
            calculate_omitted_amounts(&mut transaction),
            Err(SimplificationError::UnbalancedTransaction(
                original_transaction.clone()
            ))
        );
        assert_eq!(transaction, original_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_virtual_no_changes() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:GHI]   $3.40
  [TEST:JKL]  $-3.40
"#,
        );
        let original_transaction = transaction.clone();
        assert_eq!(calculate_omitted_amounts(&mut transaction), Ok(()));
        assert_eq!(transaction, original_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_virtual_single_commodity() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF           ; Comment 1
  [TEST:GHI]   $3.40
  [TEST:JKL]         ; Comment 2
"#,
        );
        let expected_transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20 ; Comment 1
  [TEST:GHI]   $3.40
  [TEST:JKL]  $-3.40 ; Comment 2
"#,
        );
        assert_eq!(calculate_omitted_amounts(&mut transaction), Ok(()));
        assert_eq!(transaction, expected_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_virtual_multi_commodity() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:GHI]   $3.40
  [TEST:JKL]   £5.60
  [TEST:MNO]         ; Comment
"#,
        );
        let expected_transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:GHI]   $3.40
  [TEST:JKL]   £5.60
  [TEST:MNO]  $-3.40 ; Comment
  [TEST:MNO]  £-5.60 ; Comment
"#,
        );
        assert_eq!(calculate_omitted_amounts(&mut transaction), Ok(()));
        assert_eq!(transaction, expected_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_virtual_error_unbalanced_transaction() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:GHI]   $3.40
  [TEST:JKL]  $-3.45
"#,
        );
        let original_transaction = transaction.clone();
        assert_eq!(
            calculate_omitted_amounts(&mut transaction),
            Err(SimplificationError::UnbalancedTransaction(
                original_transaction.clone()
            ))
        );
        assert_eq!(transaction, original_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_error_mixed_real_virtual_transaction() {
        // Ledger-cli will accept this input and produce a transaction that is balanced if
        // including the virtual posting but not if excluding it. We return an error instead.
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  [TEST:ABC]   $3.40
  TEST:DEF
"#,
        );
        let original_transaction = transaction.clone();
        assert_eq!(
            calculate_omitted_amounts(&mut transaction),
            Err(SimplificationError::UnbalancedTransaction(
                original_transaction.clone()
            ))
        );
        assert_eq!(transaction, original_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_unbalanced_virtual_no_changes() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:GHI]   $3.40
  [TEST:JKL]  $-3.40
  (TEST:MNO)   $5.60 ; Unbalanced virtual posting
"#,
        );
        let original_transaction = transaction.clone();
        assert_eq!(calculate_omitted_amounts(&mut transaction), Ok(()));
        assert_eq!(transaction, original_transaction);
    }

    #[test]
    fn test_calculate_omitted_amounts_error_unbalanced_virtual_posting_without_amount() {
        let mut transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  (TEST:GHI)
"#,
        );
        let original_transaction = transaction.clone();
        assert_eq!(
            calculate_omitted_amounts(&mut transaction),
            Err(SimplificationError::UnbalancedVirtualWithNoAmount(
                original_transaction.clone()
            ))
        );
        assert_eq!(transaction, original_transaction);
    }

    #[test]
    fn test_calculate_amounts_from_balances_no_change() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC     $3.40
  TEST:DEF    $-3.40
"#,
        );
        let original_transactions = transactions.clone();
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, original_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_single_commodity() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC           = $4.60 ; Comment
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC     $3.40 ; Comment
  TEST:DEF    $-3.40
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_multi_commodity() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:ABC     £3.40
  TEST:DEF    $-1.20
  TEST:DEF    £-3.40

2018-10-01 Marek Ogarek
  TEST:ABC           = £5.60 ; Comment
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:ABC     £3.40
  TEST:DEF    $-1.20
  TEST:DEF    £-3.40

2018-10-01 Marek Ogarek
  TEST:ABC     £2.20 ; Comment
  TEST:DEF    £-2.20
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_of_unbalanced_virtual() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF
  [TEST:ABC]   $3.40
  [TEST:DEF]

2018-10-01 Marek Ogarek
  (TEST:ABC)         = $5.60
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:ABC]   $3.40
  [TEST:DEF]  $-3.40

2018-10-01 Marek Ogarek
  (TEST:ABC)     $1.00
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_zero_balance_single_commodity() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC           = 0 ; Comment
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC    $-1.20 ; Comment
  TEST:DEF     $1.20
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_error_zero_balance_multi_commodity() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:ABC     £3.40
  TEST:DEF    $-1.20
  TEST:DEF    £-3.40

2018-10-01 Marek Ogarek
  TEST:ABC           = 0 ; Comment
  TEST:DEF
"#,
        );
        let error_transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC           = 0 ; Comment
  TEST:DEF
"#,
        );
        assert_eq!(
            calculate_amounts_from_balances(&mut transactions),
            Err(SimplificationError::ZeroBalanceMultipleCurrencies(
                error_transaction
            ))
        );
    }

    #[test]
    fn test_calculate_amounts_from_balances_error_unbalanced_transaction() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC   $1.20
  TEST:DEF  $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC   $1.21
  TEST:DEF         = $0.00
"#,
        );
        let error_transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC   $1.21
  TEST:DEF   $1.20
"#,
        );
        assert_eq!(
            calculate_amounts_from_balances(&mut transactions),
            Err(SimplificationError::UnbalancedTransaction(
                error_transaction
            ))
        );
    }

    #[test]
    fn test_calculate_amounts_from_balances_virtual_no_change() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:GHI]   $3.40
  [TEST:JKL]  $-3.40

2018-10-01 Marek Ogarek
  TEST:ABC     $5.60
  TEST:DEF    $-5.60
  [TEST:GHI]   $7.80
  [TEST:JKL]  $-7.80
"#,
        );
        let original_transactions = transactions.clone();
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, original_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_virtual_single_commodity() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF
  [TEST:ABC]   $3.40
  [TEST:DEF]

2018-10-01 Marek Ogarek
  [TEST:ABC]         = $5.60
  [TEST:DEF]
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:ABC]   $3.40
  [TEST:DEF]  $-3.40

2018-10-01 Marek Ogarek
  [TEST:ABC]   $1.00
  [TEST:DEF]  $-1.00
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_virtual_balancing_transaction() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF

2018-10-01 Marek Ogarek
  [TEST:ABC]         = $1.20
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  [TEST:ABC]   $0.00
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_error_virtual_balancing_transaction_fails() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF

2018-10-01 Marek Ogarek
  [TEST:ABC]         = $1.21
"#,
        );
        let error_transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  [TEST:ABC]  $0.01
"#,
        );
        assert_eq!(
            calculate_amounts_from_balances(&mut transactions),
            Err(SimplificationError::UnbalancedTransaction(
                error_transaction
            ))
        );
    }

    #[test]
    fn test_calculate_amounts_from_balances_virtual_multi_commodity() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  [TEST:ABC]   £3.40
  TEST:DEF    $-1.20
  [TEST:DEF]  £-3.40

2018-10-01 Marek Ogarek
  TEST:ABC           = £5.60 ; Comment
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  [TEST:ABC]   £3.40
  TEST:DEF    $-1.20
  [TEST:DEF]  £-3.40

2018-10-01 Marek Ogarek
  TEST:ABC     £2.20 ; Comment
  TEST:DEF    £-2.20
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_virtual_error_unbalanced_transaction() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  [TEST:ABC]   $3.40
  TEST:DEF    $-1.20
  [TEST:DEF]  $-3.40

2018-10-01 Marek Ogarek
  TEST:ABC    $-1.20
  TEST:DEF           = $0.00
"#,
        );
        let error_transaction = parse_transaction(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC    $-1.20
  TEST:DEF     $4.60
"#,
        );
        assert_eq!(
            calculate_amounts_from_balances(&mut transactions),
            Err(SimplificationError::UnbalancedTransaction(
                error_transaction
            ))
        );
    }

    #[test]
    fn test_calculate_amounts_from_balances_with_unbalanced_virtual() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF
  [TEST:ABC]   $3.40
  [TEST:DEF]
  (TEST:ABC)   $5.60

2018-10-01 Marek Ogarek
  TEST:ABC           = $11.20
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20
  [TEST:ABC]   $3.40
  [TEST:DEF]  $-3.40
  (TEST:ABC)   $5.60

2018-10-01 Marek Ogarek
  TEST:ABC     $1.00
  TEST:DEF    $-1.00
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_empty_account() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC    $-1.20  ; Account is empty
  TEST:DEF     $1.20

2018-10-01 Marek Ogarek
  TEST:ABC           = $3.40 ; Comment
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC    $-1.20  ; Account is empty
  TEST:DEF     $1.20

2018-10-01 Marek Ogarek
  TEST:ABC     $3.40 ; Comment
  TEST:DEF    $-3.40
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_unknown_account() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC           = $3.40 ; Comment
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $3.40 ; Comment
  TEST:DEF    $-3.40
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_balance_assertion() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF

2018-10-01 Marek Ogarek
  TEST:ABC     $3.40 = $4.60 ; Comment
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC     $3.40 ; Comment
  TEST:DEF    $-3.40
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_error_balance_assertion_fails() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF

2018-10-01 Marek Ogarek
  TEST:ABC     $3.41 = $4.60 ; Comment
  TEST:DEF
"#,
        );
        let error_transaction = transactions[1].clone();
        assert_eq!(
            calculate_amounts_from_balances(&mut transactions),
            Err(SimplificationError::BalanceAssertionFailed(
                error_transaction
            ))
        );
    }

    #[test]
    fn test_calculate_amounts_from_balances_zero_balance_assertion() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF

2018-10-01 Marek Ogarek
  TEST:ABC    $-1.20 = 0 ; Comment
  TEST:DEF
"#,
        );
        let expected_transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF    $-1.20

2018-10-01 Marek Ogarek
  TEST:ABC    $-1.20 ; Comment
  TEST:DEF     $1.20
"#,
        );
        assert_eq!(calculate_amounts_from_balances(&mut transactions), Ok(()));
        assert_eq!(transactions, expected_transactions);
    }

    #[test]
    fn test_calculate_amounts_from_balances_error_zero_balance_assertion_fails() {
        let mut transactions = parse_transactions(
            r#"
2018-10-01 Marek Ogarek
  TEST:ABC     $1.20
  TEST:DEF

2018-10-01 Marek Ogarek
  TEST:ABC    $-1.21 = 0 ; Comment
  TEST:DEF
"#,
        );
        let error_transaction = transactions[1].clone();
        assert_eq!(
            calculate_amounts_from_balances(&mut transactions),
            Err(SimplificationError::ZeroBalanceAssertionFailed(
                error_transaction
            ))
        );
    }
}
