// Copyright © 2020–2021 SitD <sitd0813@gmail.com>
//
// This file is subject to the terms of the MIT License.
// If a copy of the MIT License was not distributed with this file, you can obtain one at https://opensource.org/licenses/MIT.

//! LEA-CTR
//! 
//! * Example
//! ```
//! use lea::{prelude::*, Lea128Ctr};
//!
//! let key = arr![u8; 0x7A, 0xD3, 0x6A, 0x75, 0xD5, 0x5F, 0x30, 0x22, 0x09, 0x4E, 0x06, 0xF7, 0xC8, 0x97, 0xD8, 0xBB];
//! let nonce = arr![u8; 0x0C, 0x5F, 0x04, 0xE8, 0xB5, 0x12, 0x19, 0x5E, 0x74, 0xB3, 0xDE, 0x57, 0xE9, 0x70, 0x97, 0x9E];
//! let ptxt = [0x08, 0x7A, 0x83, 0xFC, 0xC1, 0x13, 0xA9, 0xF3, 0xE0, 0xE9, 0xD5, 0xAF, 0x32, 0xA2, 0xDD, 0x3A];
//! let ctxt = [0x2B, 0x73, 0x49, 0x7C, 0x4F, 0xC9, 0xEF, 0x38, 0xBE, 0x7A, 0x0B, 0xCB, 0x1A, 0xAB, 0x87, 0xA4];
//!
//! let mut lea128ctr = Lea128Ctr::new(&key, &nonce);
//!
//! // Encryption
//! let mut block = ptxt.clone();
//! lea128ctr.apply_keystream(&mut block);
//! assert_eq!(block, ctxt);
//!
//! lea128ctr.seek(0);
//!
//! // Decryption
//! let mut block = ctxt.clone();
//! lea128ctr.apply_keystream(&mut block);
//! assert_eq!(block, ptxt);
//! ```

pub use ctr::cipher;

use ctr::Ctr64BE;

use crate::{Lea128, Lea192, Lea256};

pub type Lea128Ctr = Ctr64BE<Lea128>;
pub type Lea192Ctr = Ctr64BE<Lea192>;
pub type Lea256Ctr = Ctr64BE<Lea256>;

#[cfg(test)]
mod tests {
	extern crate alloc;

	use alloc::{vec::Vec, vec};

	use crate::{prelude::*, Lea128Ctr, Lea192Ctr, Lea256Ctr};

	struct TestCase<T> where
	T: NewCipher {
		key: GenericArray<u8, T::KeySize>,
		nonce: GenericArray<u8, T::NonceSize>,
		ptxt: Vec<u8>,
		ctxt: Vec<u8>
	}

	#[test]
	fn lea128ctr() {
		let test_cases: [TestCase<Lea128Ctr>; 4] = [
			TestCase {
				key: arr![u8; 0x7A, 0xD3, 0x6A, 0x75, 0xD5, 0x5F, 0x30, 0x22, 0x09, 0x4E, 0x06, 0xF7, 0xC8, 0x97, 0xD8, 0xBB],
				nonce: arr![u8; 0x0C, 0x5F, 0x04, 0xE8, 0xB5, 0x12, 0x19, 0x5E, 0x74, 0xB3, 0xDE, 0x57, 0xE9, 0x70, 0x97, 0x9E],
				ptxt: vec![0x08, 0x7A, 0x83, 0xFC, 0xC1, 0x13, 0xA9, 0xF3, 0xE0, 0xE9, 0xD5, 0xAF, 0x32, 0xA2, 0xDD, 0x3A],
				ctxt: vec![0x2B, 0x73, 0x49, 0x7C, 0x4F, 0xC9, 0xEF, 0x38, 0xBE, 0x7A, 0x0B, 0xCB, 0x1A, 0xAB, 0x87, 0xA4]
			},
			TestCase {
				key: arr![u8; 0xAA, 0xE8, 0x41, 0xA8, 0x08, 0x58, 0x8C, 0x3B, 0x3F, 0x2F, 0xDA, 0x16, 0x9B, 0x8F, 0x96, 0x8D],
				nonce: arr![u8; 0x73, 0x8B, 0x58, 0x68, 0xC4, 0x39, 0xAC, 0x91, 0x19, 0x08, 0x15, 0x10, 0xAE, 0x54, 0x51, 0x39],
				ptxt: vec![0xD4, 0x38, 0xDF, 0x24, 0x97, 0xA4, 0xBF, 0x5E, 0x9F, 0xAF, 0x79, 0xDE, 0x54, 0x2E, 0x60, 0x3B, 0x46, 0x85, 0x18, 0x1C, 0xCB, 0xBD, 0x56, 0xEF, 0xEA, 0xD7, 0xE7, 0xDE, 0x77, 0x60, 0xF4, 0xFE, 0x80, 0x46, 0x84, 0xCA, 0xE8, 0xEA, 0x42, 0xD1, 0x52, 0x76, 0x81, 0xAB, 0x40, 0x71, 0x7D, 0x31, 0x7C, 0x91, 0xE3, 0xEE, 0xB8, 0xD0, 0x94, 0xD0, 0x71, 0xC1, 0xA6, 0x23, 0x19, 0x25, 0xAC, 0xC0],
				ctxt: vec![0x65, 0xAC, 0x04, 0x22, 0x41, 0x45, 0xF8, 0xD5, 0x26, 0xB9, 0xAA, 0x88, 0xA7, 0xBF, 0x4A, 0xCF, 0x3B, 0x47, 0x33, 0x5B, 0x5E, 0x12, 0xCE, 0x11, 0x1F, 0x1B, 0xC1, 0xF8, 0x07, 0xF4, 0x15, 0x5A, 0x97, 0xCE, 0x73, 0xEB, 0x2F, 0x5F, 0x75, 0x10, 0x4B, 0x74, 0xC2, 0xF8, 0xDF, 0xDF, 0x9A, 0xF8, 0xD6, 0x5C, 0xAF, 0x69, 0x39, 0xD9, 0xFC, 0x75, 0x33, 0xF7, 0x0F, 0xE3, 0x6F, 0xE7, 0x44, 0x78]
			},
			TestCase {
				key: arr![u8; 0x21, 0x26, 0x97, 0x77, 0x2E, 0x31, 0x23, 0x07, 0x21, 0xB4, 0xC2, 0xF3, 0x3C, 0xFE, 0x56, 0x27],
				nonce: arr![u8; 0x14, 0x5D, 0xC0, 0x8D, 0x38, 0x37, 0xA6, 0xC5, 0x13, 0x56, 0x73, 0x1C, 0x14, 0x84, 0x9F, 0xBC],
				ptxt: vec![0x0C, 0x58, 0x30, 0x20, 0xD7, 0x4F, 0x81, 0xCB, 0x3E, 0x1E, 0x42, 0x3A, 0x1C, 0x47, 0xC0, 0x17, 0xC0, 0xEB, 0x66, 0xAF, 0x93, 0xDF, 0x84, 0xA6, 0xFA, 0x02, 0x50, 0xE9, 0x7E, 0xCD, 0x2B, 0xE5, 0x2B, 0x2B, 0x25, 0xF7, 0x35, 0x8B, 0xC2, 0x23, 0xE2, 0x34, 0xFF, 0x08, 0xA3, 0xD9, 0x90, 0x13, 0x84, 0x6E, 0x6C, 0xF4, 0xC7, 0x3A, 0x8B, 0x4E, 0xCC, 0x2C, 0xEE, 0xB3, 0x35, 0x72, 0xE1, 0xCF, 0x47, 0x48, 0x7D, 0xE3, 0x92, 0x0E, 0x70, 0x76, 0xD4, 0x9E, 0x01, 0x47, 0x1B, 0xDC, 0x4E, 0x85, 0x2A, 0x8F, 0xD8, 0x42, 0x1E, 0x6F, 0x42, 0x26, 0x51, 0x7E, 0x56, 0x60, 0x7F, 0x9C, 0x49, 0xE2, 0x27, 0x56, 0x40, 0xCE, 0x34, 0x00, 0x12, 0x2B, 0xDD, 0x03, 0x50, 0xDD, 0xCA, 0x77, 0x83, 0xD4],
				ctxt: vec![0xC1, 0xCF, 0xF6, 0xC9, 0xF2, 0xBE, 0x60, 0x94, 0xA2, 0x62, 0x54, 0x65, 0xF8, 0x9C, 0x88, 0x92, 0x16, 0x52, 0xF3, 0x67, 0x04, 0xD1, 0x7C, 0x50, 0x11, 0x88, 0x06, 0x6F, 0x4D, 0xE1, 0x73, 0xF3, 0x37, 0x16, 0x7D, 0x6C, 0xE3, 0x0F, 0xD3, 0xF5, 0x74, 0x0B, 0x1E, 0xE7, 0x32, 0xB1, 0x07, 0xAF, 0x0C, 0xFB, 0xD5, 0xA0, 0x23, 0xFA, 0x44, 0x9F, 0x61, 0x25, 0xDB, 0xBD, 0x7E, 0xCA, 0x65, 0x1E, 0xE5, 0x11, 0x59, 0x48, 0xAE, 0x98, 0x64, 0x96, 0xFC, 0xF3, 0xF7, 0xF7, 0xBB, 0x4A, 0x61, 0xF1, 0x42, 0xEF, 0x9A, 0xFA, 0x92, 0xE0, 0xC4, 0x3A, 0x8D, 0x92, 0x07, 0xBB, 0xB3, 0xCD, 0xB7, 0xB4, 0x49, 0x10, 0xB4, 0x4A, 0x41, 0xE4, 0xDD, 0x6D, 0xF3, 0xF4, 0x6B, 0x70, 0x0E, 0x05, 0x65, 0xEB]
			},
			TestCase {
				key: arr![u8; 0xFC, 0xEC, 0x3E, 0x94, 0x9E, 0x90, 0xF8, 0xB5, 0x93, 0xE6, 0x97, 0x38, 0x23, 0x29, 0x36, 0x65],
				nonce: arr![u8; 0xC9, 0xF8, 0xCA, 0xE3, 0xD9, 0x64, 0xF0, 0x73, 0x65, 0x48, 0xE9, 0xDF, 0x62, 0xD9, 0xE2, 0x2C],
				ptxt: vec![0x07, 0x7D, 0x79, 0x17, 0x76, 0xE1, 0x7E, 0xC0, 0x9E, 0x45, 0xF6, 0xA0, 0x60, 0x1B, 0x66, 0xC0, 0xF0, 0xD1, 0x4E, 0x2D, 0x7F, 0xEB, 0xF3, 0xA7, 0x17, 0x54, 0x61, 0x99, 0xC6, 0xF6, 0xB1, 0x4E, 0xFE, 0x88, 0x88, 0x61, 0x3C, 0xA7, 0xE0, 0x75, 0xE8, 0x29, 0x0B, 0x27, 0x7C, 0xAE, 0xF4, 0x41, 0xE9, 0x77, 0xA9, 0x30, 0x37, 0x7C, 0x16, 0xB9, 0x6B, 0xB8, 0x13, 0xE7, 0xAD, 0xC8, 0xA2, 0x48, 0xAA, 0xB4, 0x71, 0x59, 0x38, 0x0D, 0xA7, 0x3E, 0x38, 0x38, 0xDD, 0xB6, 0xC1, 0x09, 0x69, 0x4F, 0x7B, 0x94, 0xE3, 0xD6, 0x48, 0x3F, 0xE2, 0x12, 0x2A, 0x1C, 0x07, 0xB2, 0x61, 0x76, 0x3D, 0x83, 0xD3, 0xAA, 0x3E, 0xE6, 0xB1, 0x38, 0x5A, 0x82, 0x58, 0x1A, 0x74, 0x36, 0x75, 0x55, 0x4D, 0x51, 0x6D, 0xCD, 0x05, 0x06, 0xFC, 0x5D, 0xDE, 0x1A, 0x1C, 0x27, 0x44, 0xE0, 0x28, 0x29, 0x0A, 0x67, 0x41, 0x12, 0xF7, 0xF2, 0xF1, 0x53, 0x81, 0xA8, 0x0E, 0x78, 0xD8, 0x8D, 0xE1, 0xB9, 0x26, 0xB1, 0x88, 0xCC, 0x15, 0xA8, 0x99, 0xFE, 0x93, 0x39, 0x08, 0x82, 0xD2, 0x5A, 0x4B, 0x09, 0x92, 0x5D],
				ctxt: vec![0xF8, 0x67, 0x10, 0x0F, 0x73, 0x13, 0x15, 0x94, 0xF5, 0x7F, 0x40, 0x3F, 0x5D, 0x60, 0x1A, 0x2F, 0x79, 0xCE, 0xC0, 0x86, 0x27, 0x96, 0x0D, 0xFD, 0x83, 0x01, 0x05, 0xF8, 0x13, 0x47, 0xE9, 0x9E, 0x9D, 0xE2, 0x14, 0x90, 0x75, 0xED, 0xD0, 0x92, 0x6C, 0xC8, 0x74, 0x6E, 0x2B, 0xBD, 0xAF, 0xB8, 0x7F, 0x60, 0x52, 0x75, 0x39, 0xCC, 0x24, 0xA7, 0x15, 0xEC, 0x79, 0x2F, 0x67, 0x5A, 0xCE, 0xC4, 0x13, 0x0A, 0x3F, 0x38, 0x4A, 0xE3, 0x99, 0x14, 0xC8, 0x4E, 0x14, 0xBE, 0xD7, 0x16, 0x17, 0xC1, 0xC9, 0xF4, 0xA8, 0x4A, 0x19, 0x04, 0x90, 0x48, 0x81, 0x6D, 0x3C, 0x84, 0xCE, 0x17, 0xDD, 0x27, 0xE5, 0x1C, 0x0E, 0xD0, 0x51, 0x95, 0xEA, 0x6F, 0xB5, 0xC6, 0x28, 0x18, 0x0B, 0xE9, 0xE2, 0x5D, 0xA8, 0x35, 0xDE, 0x16, 0x7A, 0x4B, 0x26, 0x59, 0x57, 0x38, 0xC8, 0xDE, 0xA6, 0x9A, 0x0A, 0x63, 0xCF, 0x92, 0x2F, 0x49, 0xB3, 0x68, 0xB3, 0x25, 0xA4, 0x16, 0x61, 0xAF, 0xB4, 0xFD, 0x9E, 0xB3, 0xF0, 0xB6, 0x7B, 0x53, 0xD1, 0x86, 0xCA, 0x6A, 0x1E, 0xF5, 0x92, 0x5D, 0x22, 0x0D, 0x0F, 0x70]
			}
		];

		for test_case in test_cases {
			let mut lea128ctr = Lea128Ctr::new(&test_case.key, &test_case.nonce);

			// Encryption
			let mut block = test_case.ptxt.clone();
			lea128ctr.apply_keystream(&mut block);
			assert_eq!(block, test_case.ctxt);

			// Decryption
			lea128ctr.seek(0);
			let mut block = test_case.ctxt.clone();
			lea128ctr.apply_keystream(&mut block);
			assert_eq!(block, test_case.ptxt);
		}
	}

	#[test]
	fn lea192ctr() {
		let test_cases: [TestCase<Lea192Ctr>; 4] = [
			TestCase {
				key: arr![u8; 0xBB, 0x93, 0xA2, 0x64, 0x3E, 0x84, 0xA4, 0x1A, 0x23, 0xFA, 0x12, 0xA5, 0x4D, 0x5E, 0x7E, 0xD6, 0x94, 0x39, 0x1E, 0xA3, 0x68, 0x49, 0x87, 0xD8],
				nonce: arr![u8; 0xB7, 0xD5, 0xB9, 0x09, 0x11, 0x3D, 0x5C, 0xCB, 0x0B, 0xD5, 0x49, 0x24, 0xE1, 0xF3, 0x4C, 0x3F],
				ptxt: vec![0x5F, 0x47, 0x28, 0x64, 0x01, 0x6B, 0xDC, 0x28, 0x59, 0xBB, 0x25, 0xE1, 0xB1, 0x67, 0x44, 0x5D],
				ctxt: vec![0xC6, 0x35, 0x7A, 0xBD, 0x1D, 0x38, 0x24, 0xF2, 0xC7, 0x2E, 0xD6, 0xEF, 0x4B, 0x76, 0xD8, 0x97]
			},
			TestCase {
				key: arr![u8; 0x8F, 0xD2, 0x38, 0x26, 0xEA, 0x26, 0x49, 0x10, 0x1C, 0x05, 0x90, 0xCA, 0x7D, 0x73, 0xD6, 0x8D, 0xB4, 0x89, 0x9A, 0x6B, 0xDB, 0x10, 0x56, 0x86],
				nonce: arr![u8; 0xDE, 0x4E, 0x04, 0x80, 0x24, 0x20, 0xE1, 0x32, 0x72, 0x49, 0x71, 0xA1, 0x0C, 0x0A, 0xC3, 0x2D],
				ptxt: vec![0x1D, 0xF3, 0xF6, 0x0C, 0x58, 0xF8, 0xFC, 0xF7, 0xE0, 0xC7, 0x9E, 0x23, 0x65, 0x59, 0xC0, 0x73, 0x90, 0xC7, 0xE7, 0xEC, 0xA2, 0x60, 0xF8, 0x69, 0x58, 0xF8, 0x43, 0x90, 0x11, 0x82, 0xBF, 0x04, 0x32, 0xE2, 0x97, 0xDC, 0xCA, 0xFC, 0xE6, 0x55, 0x72, 0x11, 0xBF, 0xC3, 0x78, 0x4B, 0x71, 0xCE, 0x3C, 0x96, 0x09, 0xDA, 0xD9, 0xB2, 0x18, 0xC7, 0x08, 0x86, 0xB6, 0xDB, 0x42, 0xB8, 0xC6, 0xFC],
				ctxt: vec![0x43, 0xA1, 0x5C, 0x83, 0x7A, 0x18, 0x06, 0x35, 0xCE, 0xA9, 0x49, 0x45, 0xC5, 0xA5, 0x9D, 0x44, 0xA0, 0xFC, 0x0D, 0x3B, 0xB7, 0xC1, 0x4E, 0xF0, 0x07, 0xCC, 0x40, 0x42, 0xC5, 0x76, 0x4A, 0x90, 0xA5, 0x68, 0xBD, 0xAC, 0x6D, 0xD8, 0x60, 0x1E, 0xD4, 0x37, 0xFA, 0x1A, 0x45, 0x7E, 0x85, 0x86, 0xF9, 0x3C, 0x5E, 0x4E, 0x03, 0x54, 0xC7, 0x8C, 0x2E, 0x98, 0xDA, 0x31, 0x17, 0xF4, 0x8F, 0x0A]
			},
			TestCase {
				key: arr![u8; 0x87, 0xD7, 0xD9, 0xB0, 0x85, 0x28, 0x4E, 0x5F, 0x91, 0xF8, 0x3C, 0xB7, 0x63, 0x36, 0xAB, 0x2F, 0xE5, 0x2E, 0x84, 0xA1, 0xDD, 0xA2, 0x51, 0xFA],
				nonce: arr![u8; 0xCD, 0x1D, 0x45, 0x36, 0x17, 0x84, 0x7C, 0x8C, 0x65, 0xEE, 0x83, 0xE5, 0x48, 0xB5, 0x1B, 0x93],
				ptxt: vec![0x20, 0xCE, 0x3A, 0x40, 0x3B, 0x55, 0x53, 0x24, 0x41, 0x6C, 0xD7, 0x7C, 0xCF, 0x46, 0xFC, 0x37, 0x62, 0x7F, 0xBD, 0xF2, 0x77, 0xAF, 0x22, 0x26, 0xF0, 0x3E, 0xC1, 0xA0, 0xBA, 0x7A, 0x85, 0x32, 0xAD, 0xE6, 0xAE, 0xA9, 0xB3, 0xD5, 0x19, 0xFE, 0x2D, 0x38, 0xC2, 0xD1, 0x92, 0xB5, 0x87, 0x12, 0x59, 0xF9, 0x2F, 0x02, 0x1A, 0x4E, 0xAA, 0x5A, 0xF0, 0x70, 0x9A, 0xC9, 0x21, 0x9C, 0x13, 0xA4, 0x01, 0xEB, 0x9F, 0xDA, 0x13, 0xDE, 0x86, 0x26, 0x72, 0x3A, 0x4C, 0x88, 0x6F, 0x16, 0x7A, 0xF5, 0x7C, 0x32, 0xA2, 0x4E, 0x49, 0x8A, 0x9D, 0x8F, 0x2C, 0x2B, 0x17, 0x49, 0xC6, 0x46, 0x4E, 0x52, 0xE4, 0x83, 0x16, 0xBA, 0xA3, 0x97, 0x21, 0x02, 0xD8, 0x19, 0x7D, 0x89, 0xAE, 0x91, 0x5E, 0x47],
				ctxt: vec![0xBE, 0x76, 0x4F, 0xEB, 0x48, 0x8B, 0x16, 0x28, 0xF4, 0xC5, 0xFE, 0xFA, 0x71, 0x98, 0x23, 0xFA, 0x64, 0xD7, 0xA7, 0xC6, 0x00, 0x52, 0x18, 0xCC, 0xE0, 0x70, 0xB5, 0x15, 0x31, 0x4E, 0x2C, 0x66, 0xDA, 0x1B, 0xB1, 0xA7, 0x6B, 0x87, 0x87, 0x90, 0x7B, 0x10, 0xB7, 0xB6, 0xAA, 0x00, 0xA5, 0x54, 0x85, 0x54, 0xC9, 0xA1, 0x47, 0x1C, 0x78, 0x2D, 0x92, 0x69, 0xC8, 0x86, 0x78, 0x4E, 0xF4, 0x1F, 0xB6, 0x50, 0x3B, 0x90, 0x08, 0x7A, 0xC5, 0x23, 0xD8, 0x14, 0xB9, 0xDA, 0x4E, 0xDD, 0x47, 0x30, 0x55, 0xF4, 0x90, 0x48, 0x5F, 0x60, 0xD0, 0x66, 0x6B, 0xB2, 0xC5, 0x64, 0xB0, 0x90, 0x98, 0x66, 0xBB, 0x36, 0x9B, 0xB5, 0x25, 0xE8, 0x1F, 0x46, 0x67, 0xBE, 0xB5, 0x55, 0x4A, 0x11, 0x77, 0xE0]
			},
			TestCase {
				key: arr![u8; 0x84, 0x1E, 0xCA, 0x09, 0x74, 0xEE, 0xC0, 0x3A, 0xE8, 0xBD, 0x0F, 0x57, 0xB8, 0x16, 0xEB, 0x4F, 0x69, 0x79, 0xA3, 0xCA, 0x51, 0xF2, 0xDE, 0x60],
				nonce: arr![u8; 0xFC, 0xF0, 0x24, 0x08, 0xCF, 0x55, 0xA1, 0xD3, 0xEB, 0xCA, 0x26, 0xDA, 0x55, 0x55, 0x71, 0x74],
				ptxt: vec![0x53, 0x2D, 0xAE, 0xAD, 0x19, 0xCD, 0x3E, 0xF4, 0xA4, 0x47, 0xB6, 0x14, 0xE7, 0xDB, 0x2B, 0x66, 0x25, 0xC8, 0xAD, 0x44, 0x9E, 0x62, 0x11, 0xC0, 0x6D, 0x65, 0xF4, 0x96, 0xB1, 0x89, 0xFC, 0x60, 0xEB, 0x56, 0x61, 0x09, 0xA7, 0x3A, 0xAC, 0x84, 0x5F, 0xD9, 0xBF, 0xBE, 0x9C, 0xA4, 0x16, 0xD1, 0x5E, 0xAD, 0x4C, 0x7A, 0xBE, 0xB9, 0xE1, 0xCD, 0xD2, 0x97, 0x3A, 0x27, 0xD1, 0xB1, 0xE9, 0x65, 0x77, 0xE1, 0x2F, 0x53, 0xAB, 0x86, 0xBF, 0x67, 0x60, 0xD6, 0xC5, 0xB0, 0xB9, 0x76, 0x27, 0x09, 0x70, 0x48, 0x0B, 0x92, 0x78, 0x84, 0x99, 0x61, 0xE1, 0x0A, 0x02, 0x74, 0xFD, 0xF6, 0xC1, 0xEA, 0xC1, 0x75, 0x21, 0x73, 0x6D, 0xD8, 0xFF, 0x06, 0x70, 0xE7, 0xD1, 0xD2, 0x85, 0x78, 0xE7, 0x76, 0x23, 0x40, 0xF1, 0x74, 0x14, 0xE8, 0xC2, 0xE3, 0x63, 0x63, 0x53, 0x65, 0x7C, 0x80, 0x0B, 0x59, 0x8F, 0xBB, 0x3D, 0x52, 0x35, 0x59, 0xF3, 0xC7, 0x56, 0xB4, 0xEA, 0x0C, 0x4A, 0xD3, 0xDD, 0x80, 0x3E, 0x3D, 0x06, 0x09, 0xDA, 0x0F, 0xE3, 0xBD, 0x21, 0x4D, 0x36, 0xE2, 0x98, 0x76, 0x4F, 0x19],
				ctxt: vec![0x3E, 0x23, 0xF2, 0x14, 0x9F, 0x53, 0xE8, 0x64, 0xD3, 0x4E, 0x6A, 0xBD, 0xA7, 0xAD, 0xF9, 0xA3, 0x80, 0x5F, 0x27, 0x75, 0x2E, 0xEE, 0xCC, 0xDA, 0x72, 0x07, 0x41, 0x99, 0x1D, 0x37, 0x34, 0x3B, 0x00, 0xFD, 0x35, 0x03, 0x06, 0xF3, 0xBA, 0xD8, 0xA8, 0xC0, 0x31, 0x0C, 0x7F, 0x96, 0x1F, 0xCF, 0x46, 0x96, 0x4E, 0x38, 0x93, 0x90, 0xD0, 0xFC, 0xCA, 0x59, 0x1F, 0xE0, 0x5D, 0xC4, 0x9B, 0x48, 0x8D, 0xD2, 0xB4, 0x29, 0x18, 0xFD, 0xAD, 0x89, 0x3A, 0xCF, 0x2F, 0xA2, 0x29, 0x59, 0xC6, 0xC5, 0x91, 0x0C, 0xB7, 0xE5, 0x7A, 0x1E, 0xC7, 0xC1, 0x07, 0x88, 0x90, 0xA1, 0xB3, 0xA3, 0x94, 0x41, 0x56, 0x7E, 0x03, 0x6D, 0x3B, 0x90, 0x0A, 0x83, 0xED, 0x40, 0xB4, 0xD7, 0x83, 0x61, 0xCD, 0xB5, 0xF2, 0xB7, 0x83, 0xBC, 0x1A, 0x0A, 0x41, 0x6D, 0xAB, 0xCA, 0xDB, 0xD8, 0xDE, 0xD4, 0x4A, 0x76, 0xF7, 0x3A, 0xE2, 0x35, 0x76, 0x3B, 0x6E, 0x8C, 0xED, 0xC2, 0x37, 0xB4, 0x32, 0x9F, 0x71, 0x62, 0x4E, 0x55, 0xDC, 0x42, 0xAE, 0xC5, 0xB3, 0x80, 0xD8, 0x04, 0x20, 0xF2, 0x85, 0x94, 0xE6, 0xB3]
			}
		];

		for test_case in test_cases {
			let mut lea192ctr = Lea192Ctr::new(&test_case.key, &test_case.nonce);

			// Encryption
			let mut block = test_case.ptxt.clone();
			lea192ctr.apply_keystream(&mut block);
			assert_eq!(block, test_case.ctxt);

			// Decryption
			lea192ctr.seek(0);
			let mut block = test_case.ctxt.clone();
			lea192ctr.apply_keystream(&mut block);
			assert_eq!(block, test_case.ptxt);
		}
	}

	#[test]
	fn lea256ctr() {
		let test_cases: [TestCase<Lea256Ctr>; 4] = [
			TestCase {
				key: arr![u8; 0xAA, 0x5B, 0x8D, 0xD6, 0x4B, 0x30, 0x23, 0x13, 0xDC, 0xE4, 0x18, 0x46, 0x4E, 0xAE, 0x92, 0x90, 0x8B, 0xE9, 0x53, 0x37, 0x11, 0x21, 0x84, 0x56, 0xE0, 0x6E, 0xB1, 0xD3, 0x97, 0x00, 0x16, 0x92],
				nonce: arr![u8; 0xDA, 0xFC, 0x19, 0xE8, 0xF6, 0x87, 0x17, 0x53, 0xC8, 0x1F, 0x63, 0x68, 0xDB, 0x32, 0x8C, 0x0C],
				ptxt: vec![0xD0, 0xE9, 0xDF, 0xE7, 0x03, 0x45, 0x2D, 0x16, 0x6B, 0x6E, 0xCF, 0x20, 0xC2, 0x48, 0xE6, 0x2C],
				ctxt: vec![0xFC, 0x9A, 0x78, 0xBA, 0x8F, 0x08, 0xAE, 0xA8, 0x2F, 0x9A, 0x37, 0xE5, 0xBD, 0x2C, 0x04, 0xD8]
			},
			TestCase {
				key: arr![u8; 0x98, 0xB3, 0x19, 0x75, 0xA9, 0x65, 0x95, 0x8A, 0x0E, 0x88, 0x79, 0x3C, 0x1D, 0x2B, 0x64, 0x3E, 0x0F, 0xAF, 0x05, 0x12, 0xAA, 0xD9, 0xDE, 0xC4, 0x1D, 0x25, 0x48, 0xD8, 0x2B, 0x5D, 0xB8, 0x81],
				nonce: arr![u8; 0x8B, 0x6B, 0x40, 0x14, 0x8F, 0x6C, 0x59, 0xBB, 0x05, 0x05, 0xDF, 0x6E, 0x5B, 0xFA, 0x5D, 0x1F],
				ptxt: vec![0x06, 0xFC, 0x8B, 0x35, 0x23, 0xC3, 0x15, 0x3C, 0x5F, 0x5D, 0x9E, 0xDB, 0x15, 0xC6, 0x05, 0x06, 0xB7, 0xB8, 0xE7, 0x74, 0x6F, 0xC4, 0x65, 0x54, 0x06, 0xA2, 0x27, 0x3C, 0x04, 0xC6, 0xA0, 0x62, 0x19, 0x33, 0x94, 0x0E, 0xBB, 0x94, 0xD8, 0x4F, 0x11, 0x89, 0x5A, 0xEE, 0x10, 0x3F, 0x61, 0xA0, 0xE4, 0x43, 0x15, 0x7F, 0x91, 0x98, 0x41, 0xBB, 0xDA, 0x07, 0x58, 0x8D, 0x62, 0xB6, 0xB6, 0x6D],
				ctxt: vec![0x8F, 0x7F, 0x09, 0x50, 0x8A, 0xA0, 0xC2, 0x64, 0x0E, 0xEB, 0xC8, 0x1A, 0x87, 0x53, 0xBC, 0xBB, 0xDA, 0xAB, 0x75, 0x1E, 0xD7, 0x04, 0x0B, 0x20, 0x7A, 0x64, 0x0F, 0xC9, 0x6C, 0xDE, 0xB9, 0xB4, 0x15, 0xBC, 0x1E, 0x60, 0x3B, 0xFC, 0x40, 0x50, 0xDE, 0x8A, 0xAD, 0x14, 0xDE, 0x8F, 0x82, 0x4F, 0x3C, 0xBB, 0xC6, 0x5A, 0x51, 0x69, 0x2F, 0x13, 0xAA, 0x4B, 0xBA, 0x5D, 0xB5, 0x57, 0xBD, 0xF0]
			},
			TestCase {
				key: arr![u8; 0x81, 0x3F, 0xA2, 0x4F, 0xBA, 0x5A, 0x46, 0xCD, 0x44, 0xEC, 0x3B, 0x2B, 0xAE, 0x5E, 0x19, 0x35, 0xE9, 0xE8, 0x07, 0x05, 0xF1, 0xA8, 0x7D, 0xBD, 0xAF, 0xF2, 0x4F, 0x7C, 0x1F, 0xE8, 0xE7, 0x8A],
				nonce: arr![u8; 0xAE, 0x4A, 0xD1, 0xD6, 0xE6, 0x8F, 0x7E, 0xA3, 0x4C, 0x34, 0x41, 0x7E, 0x89, 0x54, 0x9E, 0x73],
				ptxt: vec![0xC9, 0x79, 0xC4, 0x80, 0x61, 0xB2, 0x58, 0x4A, 0xB4, 0xE5, 0x74, 0x0F, 0x57, 0x67, 0xF0, 0xDC, 0x73, 0xCA, 0xE0, 0x00, 0x6B, 0xF6, 0x5D, 0xC0, 0xBF, 0x7B, 0x89, 0x4C, 0x31, 0x26, 0xCD, 0xF2, 0x26, 0xD3, 0x65, 0x92, 0x4E, 0x81, 0x1F, 0x52, 0x87, 0xAC, 0x5F, 0x92, 0xFF, 0xD7, 0x67, 0x23, 0x99, 0x68, 0xD5, 0xB3, 0x93, 0xB8, 0x6D, 0x8C, 0x65, 0x6B, 0x19, 0x7F, 0xEC, 0xFD, 0x2E, 0x1A, 0xC7, 0x9E, 0xF1, 0x22, 0x02, 0x40, 0x5A, 0x3D, 0xF1, 0xEE, 0x18, 0xEE, 0x5F, 0x5F, 0xD4, 0xC6, 0xE9, 0xCB, 0xBA, 0xD9, 0xA4, 0xFD, 0x36, 0x70, 0x06, 0xAA, 0xFB, 0xFD, 0x03, 0x01, 0x4A, 0x54, 0x76, 0x83, 0x71, 0x17, 0xC3, 0x15, 0x92, 0x72, 0xBA, 0x53, 0xA5, 0x09, 0xBF, 0x28, 0xC0, 0x2F],
				ctxt: vec![0xC9, 0xF7, 0xBA, 0x69, 0xDA, 0xAA, 0xF8, 0xBA, 0x4C, 0x4A, 0xAF, 0xA9, 0x17, 0x87, 0xE1, 0x5F, 0xA0, 0x3F, 0x17, 0x64, 0x07, 0xA2, 0x64, 0x04, 0x5D, 0xD6, 0x55, 0xF2, 0x72, 0x3B, 0x26, 0x0D, 0x42, 0xBD, 0xAC, 0xE6, 0xD2, 0x26, 0xC3, 0x78, 0x34, 0xF2, 0xA3, 0xAE, 0x2A, 0x1D, 0x3B, 0xF5, 0x07, 0x1F, 0x72, 0xDD, 0x67, 0x70, 0xA4, 0x4C, 0x40, 0x92, 0x73, 0x5D, 0x35, 0x05, 0xED, 0x44, 0x05, 0xDE, 0xB9, 0xD4, 0x51, 0x92, 0xC6, 0xF8, 0x3D, 0x83, 0xE3, 0xE2, 0x46, 0x3C, 0x1C, 0x63, 0xD8, 0xB8, 0x35, 0xBE, 0x81, 0x0A, 0x6D, 0x8E, 0xA9, 0x2F, 0x5C, 0xAC, 0x71, 0x99, 0x01, 0x68, 0xF3, 0x94, 0x8D, 0x1C, 0x95, 0x6C, 0x86, 0x51, 0xCD, 0xA0, 0xD9, 0x2E, 0x46, 0x3C, 0x81, 0xFF]
			},
			TestCase {
				key: arr![u8; 0xEB, 0xE8, 0xEE, 0x96, 0x66, 0xD0, 0x6D, 0xB7, 0x69, 0xCD, 0xA8, 0xB9, 0x8F, 0x1E, 0xAB, 0x04, 0xE7, 0xA6, 0xA4, 0xA8, 0x99, 0xFB, 0x9F, 0x05, 0xCD, 0xBB, 0x95, 0xCB, 0xC8, 0x1F, 0xA5, 0x26],
				nonce: arr![u8; 0x58, 0xD2, 0xA1, 0x32, 0x73, 0x03, 0xCC, 0xB5, 0x1B, 0xB9, 0xE2, 0x0D, 0x84, 0x66, 0x59, 0x67],
				ptxt: vec![0x79, 0xC0, 0xE7, 0x32, 0xFC, 0xCC, 0x44, 0xD4, 0x2D, 0x3B, 0x31, 0x9B, 0x6D, 0xFA, 0xB9, 0xF6, 0xC2, 0x05, 0xB7, 0xE5, 0x7D, 0x7C, 0x98, 0xAE, 0x1B, 0xF8, 0x62, 0xD2, 0x6A, 0x1F, 0xF5, 0x3F, 0xED, 0x76, 0x92, 0xC7, 0x80, 0x77, 0x99, 0xD1, 0x3F, 0xE4, 0x97, 0x4E, 0xA5, 0x5A, 0x7F, 0xEF, 0xF1, 0x29, 0x38, 0x95, 0xCE, 0x63, 0x58, 0x0A, 0x32, 0x33, 0x30, 0xEE, 0x87, 0x70, 0x08, 0xF4, 0x09, 0x72, 0xAB, 0x4E, 0x6F, 0x25, 0x27, 0x65, 0xCD, 0x5B, 0xCE, 0xCE, 0xB9, 0x67, 0x80, 0x79, 0xAD, 0xE7, 0x2D, 0x2C, 0xAC, 0xE1, 0x95, 0x30, 0x28, 0x12, 0x52, 0x4B, 0x24, 0x82, 0x19, 0xEE, 0x96, 0x5C, 0x3D, 0xAE, 0x0F, 0xFD, 0x74, 0xF8, 0x9D, 0x4B, 0xDE, 0x01, 0xF1, 0x48, 0x43, 0xFD, 0xBD, 0xE7, 0x9D, 0x91, 0x60, 0x1E, 0xD6, 0x8A, 0xC5, 0x3C, 0xD2, 0xCF, 0x88, 0x7D, 0xB0, 0x94, 0x5B, 0xDB, 0x4D, 0xD1, 0xA9, 0x28, 0x0A, 0xF3, 0x79, 0x5A, 0xD0, 0xD1, 0x94, 0x26, 0x51, 0xE1, 0xEA, 0xD0, 0x90, 0xAC, 0x32, 0x41, 0xA3, 0x7F, 0xD1, 0x5A, 0xB7, 0x64, 0xFD, 0x88, 0x56, 0x50],
				ctxt: vec![0xCA, 0xDD, 0x51, 0xE5, 0xBF, 0x4A, 0x97, 0x8F, 0x79, 0x7A, 0x1C, 0x0A, 0x63, 0x0B, 0x2F, 0xC4, 0x67, 0x40, 0x0D, 0x77, 0x44, 0x30, 0x3C, 0x87, 0x3D, 0xBE, 0x2B, 0x52, 0xB1, 0xE3, 0x13, 0x7C, 0xD3, 0x6B, 0xA5, 0x23, 0x2A, 0x5E, 0xD3, 0x32, 0xB0, 0x2F, 0x20, 0xAD, 0x25, 0x76, 0xBA, 0x76, 0x2E, 0xC1, 0x66, 0x18, 0xEC, 0x4E, 0xC8, 0x1A, 0x33, 0x4B, 0x20, 0x1A, 0x0A, 0x24, 0x41, 0x38, 0x5C, 0xB9, 0xA9, 0x33, 0x5E, 0x91, 0x4F, 0xCD, 0x1E, 0x00, 0x0B, 0x8C, 0x61, 0x04, 0x07, 0x7F, 0x57, 0x4C, 0x21, 0xC0, 0x61, 0x82, 0x57, 0x1D, 0x69, 0x34, 0xA4, 0x7B, 0x93, 0xF2, 0x7A, 0x86, 0xD2, 0x0B, 0x0B, 0x7B, 0xA6, 0xAC, 0xBB, 0x7B, 0x0D, 0x56, 0x24, 0x31, 0x0A, 0x82, 0x81, 0x58, 0xC1, 0xF3, 0x36, 0xCA, 0x04, 0xA0, 0xFA, 0x01, 0xA6, 0x45, 0x1F, 0x0E, 0x87, 0x69, 0x33, 0xE5, 0x4C, 0xDC, 0x32, 0x89, 0x4A, 0xB2, 0xD3, 0x9B, 0x23, 0x2C, 0x30, 0x16, 0x38, 0xAB, 0xE0, 0xBF, 0x50, 0xCE, 0x33, 0x34, 0x45, 0x88, 0xD0, 0xA7, 0x31, 0xBF, 0x31, 0xDB, 0x42, 0x7F, 0xE2, 0x76]
			}
		];

		for test_case in test_cases {
			let mut lea256ctr = Lea256Ctr::new(&test_case.key, &test_case.nonce);

			// Encryption
			let mut block = test_case.ptxt.clone();
			lea256ctr.apply_keystream(&mut block);
			assert_eq!(block, test_case.ctxt);

			// Decryption
			lea256ctr.seek(0);
			let mut block = test_case.ctxt.clone();
			lea256ctr.apply_keystream(&mut block);
			assert_eq!(block, test_case.ptxt);
		}
	}
}