use serde::Deserialize;

use crate::types::Int;

use super::{entity_instance::EntityInstance, tile_instance::TileInstance};

#[derive(Clone, Deserialize)]
pub struct LayerInstance {
    /// Grid-based height
    #[serde(alias = "__cHei")]
    pub c_hei: Int,

    /// Grid-based width
    #[serde(alias = "__cWid")]
    pub c_wid: Int,

    /// Grid size
    #[serde(alias = "__gridSize")]
    pub grid_size: Int,

    /// Layer definition identifier
    #[serde(alias = "__identifier")]
    pub identifier: String,

    /// Layer opacity as Float [0-1]
    #[serde(alias = "__opacity")]
    pub opacity: Int,

    /// Total layer X pixel offset, including both instance and definition offsets.
    #[serde(alias = "__pxTotalOffsetX")]
    pub px_total_offset_x: Int,

    /// Total layer Y pixel offset, including both instance and definition offsets.
    #[serde(alias = "__pxTotalOffsetY")]
    pub px_total_offset_y: Int,

    /// The definition UID of corresponding Tileset, if any.
    #[serde(alias = "__tilesetDefUid")]
    pub tileset_def_uid: Option<Int>,

    /// The relative path to corresponding Tileset, if any.
    #[serde(alias = "__tilesetRelPath")]
    pub tileset_rel_path: Option<String>,

    /// Layer type (possible values: IntGrid, Entities, Tiles or AutoLayer)
    #[serde(alias = "__type")]
    pub layer_type: String,

    /// An array containing all tiles generated by Auto-layer rules. The array is already sorted in display order (ie. 1st tile is beneath 2nd, which is beneath 3rd etc.).
    /// Note: if multiple tiles are stacked in the same cell as the result of different rules, all tiles behind opaque ones will be discarded.
    #[serde(alias = "autoLayerTiles")]
    pub auto_layer_tiles: Vec<TileInstance>,

    #[serde(alias = "entityInstances")]
    pub entity_instances: Vec<EntityInstance>,

    #[serde(alias = "gridTiles")]
    pub grid_tiles: Vec<TileInstance>,

    /// Unique layer instance identifier
    pub iid: String,

    /// A list of all values in the IntGrid layer, stored in CSV format (Comma Separated Values).
    /// Order is from left to right, and top to bottom (ie. first row from left to right, followed by second row, etc).
    /// 0 means "empty cell" and IntGrid values start at 1.
    /// The array size is __cWid x __cHei cells.
    #[serde(alias = "intGridCsv")]
    pub int_grid_csv: Vec<Int>,

    /// Reference the Layer definition UID
    #[serde(alias = "layerDefUid")]
    pub layer_def_uid: Int,

    /// Reference to the UID of the level containing this layer instance
    #[serde(alias = "levelId")]
    pub level_id: Int,

    /// This layer can use another tileset by overriding the tileset UID here.
    #[serde(alias = "overrideTilesetUid")]
    pub override_tileset_uid: Option<Int>,

    /// X offset in pixels to render this layer, usually 0 (IMPORTANT: this should be added to the LayerDef optional offset, see __pxTotalOffsetX)
    #[serde(alias = "pxOffsetX")]
    pub px_offset_x: Int,

    /// Y offset in pixels to render this layer, usually 0 (IMPORTANT: this should be added to the LayerDef optional offset, see __pxTotalOffsetY)
    #[serde(alias = "pxOffsetY")]
    pub px_offset_y: Int,

    /// Layer instance visibility
    pub visible: bool,
}
