# lazer-rs

Console printer with a fluent API, written in Rust.

# lazer

Lazer is a utility for printing to the console using a fluent API. Ported to Rust from the original TypeScript package.

## Description
Lazer helps you build, format and print complex messages to the console using an expressive fluent API.

## Usage

### Simple Example

#### **`main.rs`**
```rust
use lazer::{lazer};

fn main() {
    lazer()
        .print("Hello,")
        .print_space()
        .print_green("Green World")
        .print_ln("!");
}
```

```bash
$ cargo run
Hello, Green World!

```

### Complex Example

```typescript
use lazer::{lazer};

fn main() {
    let remote_addr = "127.0.0.1";
    let method = "GET";
    let path = "/a/really/really/really/long/path/here";
    let status = 200;
    let time_ms = 20;
    let size_bytes_string = "1.10kB";

    lazer()
        .print("[").print_utc_time().print("]")
        .print_space(1).print("-").print_space(1)
        .print_pad_right(remote_addr, 15, "_")
        .print_space(2)
        .print_pad_right(method, 4, "_")
        .print_space(2)
        .print_pad_right(path, 20, "_")
        .print_space(2)
        .iff(status >= 200 && status < 300)
            .print_green(&status.to_string())
        .eliff(status >= 300 && status < 400)
            .print_yellow(&status.to_string())
        .eliff(status >= 400)
            .print_red(&status.to_string())
        .end()
        .print_space(2)
        .print_pad_right(&format!("{}ms", time_ms), 6, "_")
        .print_space(2)
        .print_ln(size_bytes_string);
}
```

```bash
$ deno run example.ts
[] - 127.0.0.1_______  GET__  /a/really/really/r+18  200  20ms___  1.10kB

```

### Buffering Example

```typescript
import { lazer } from "https://deno.land/x/lazer/mod.ts"

const getLinePrefix = () => 
{
    return lazer().buffer()
        .print_yellow('[').print_yellow("Line Prefix").print_yellow(']')
        .print_space().print("-").print_space()
        .print_yellow('[').set_color_yellow().print_utc_time().print_yellow(']')
        .print_space().print("-").print_space()
        .return();
}

lazer()
    .print(getLinePrefix())
    .print_yellow_ln("This is a prefixed line of text output");

lazer()
    .print(getLinePrefix())
    .print_yellow_ln("This is another prefixed line of text output");
```

```bash
$ deno run example.ts
[Line Prefix] - [Mon, 10 May 2021 16:31:29 GMT] - This is a prefixed line of text output
[Line Prefix] - [Mon, 10 May 2021 16:31:29 GMT] - This is another prefixed line of text output

```

### Buffer Aliasing Example

```typescript
import { lazer } from "https://deno.land/x/lazer/mod.ts"

lazer().buffer()
    .set_color_red().print_ln("Some red output to buffer")
    .store('i am an alias');

lazer().buffer()
    .load('i am an alias')
    .print_b();
```

```bash
$ deno run example.ts
Some red output to buffer

```

## Supported Platforms

### Deno
```typescript
import { lazer } from "https://deno.land/x/lazer/mod.ts"
```

### Node.js
```bash
npm i --save lazer-js
```
```javascript
const { lazer } = require('lazer-js');
```