use std::env;
use std::fs;
use std::io::prelude::*;
use std::io::{stdin, stdout, BufReader, Lines, Write};
use std::path::{Path, PathBuf};
use std::sync::{Arc, Mutex};
use std::thread;

use termion::event::{Event, Key, MouseEvent};
use termion::input::{MouseTerminal, TermRead};
use termion::raw::IntoRawMode;
use termion::screen::*;
use termion::terminal_size;

use syntect::easy::HighlightLines;
use syntect::highlighting::{Style, Theme, ThemeSet};
use syntect::parsing::{SyntaxReference, SyntaxSet};
use syntect::util::as_24_bit_terminal_escaped;

struct EditorConfiguration<'a, 'b> {
    syntax: &'b SyntaxReference,
    theme: &'a Theme,
}

struct EditorStatus {
    width: u16,
    height: u16,
    display_begin_row: usize,
    display_end_row: usize,
    cursor_row: u16,
    cursor_col: u16,
    saved: bool,
}

struct FileInformation {
    file_path: PathBuf,
    file_name: String,
    contents: Vec<String>,
}

struct Editor<'a, 'b> {
    edit_configuration: EditorConfiguration<'a, 'b>,
    edit_status: EditorStatus,
    file_information: FileInformation,
}

impl Editor<'_, '_> {
    fn load_file(&mut self) {
        let file = fs::read_to_string(&self.file_information.file_path).unwrap();
        let mut highlighter = HighlightLines::new(
            &self.edit_configuration.syntax,
            &self.edit_configuration.theme,
        );
        let ranges: Vec<(Style, &str)> =
            highlighter.highlight(file.as_str(), &SyntaxSet::load_defaults_newlines());
        let escaped = as_24_bit_terminal_escaped(&ranges[..], true);
        let split_string = escaped.lines();
        let highlighted_lines: Vec<String> = split_string.map(|s| s.to_string()).collect();
        self.file_information.contents = highlighted_lines.clone();
    }
}

fn get_file_name() -> String {
    let args: Vec<String> = env::args().collect();
    match args.len() {
        1 => String::new(),
        2 => String::from(&args[1]),
        _ => {
            print_help();
            String::new()
        }
    }
}

fn print_help() {
    println!("Usage: edit [OPTIONS] [FILE]\n");
    println!("Option\tLong\tMeaning");
}

fn create_editor_ui() -> termion::screen::AlternateScreen<
    termion::input::MouseTerminal<termion::raw::RawTerminal<std::io::Stdout>>,
> {
    let mut screen = create_screen_overlay();
    write!(screen, "{}", termion::cursor::Goto(1, 1)).unwrap();
    write!(screen, "Edit").unwrap();
    screen.flush().unwrap();
    screen
}

fn create_screen_overlay() -> termion::screen::AlternateScreen<
    termion::input::MouseTerminal<termion::raw::RawTerminal<std::io::Stdout>>,
> {
    let raw_terminal = stdout().into_raw_mode().unwrap();
    let with_mouse_support = MouseTerminal::from(raw_terminal);
    let screen = AlternateScreen::from(with_mouse_support);
    screen
}

fn display_file(editor: &mut Editor, screen: &mut dyn Write) {
    let editor_status = &mut editor.edit_status;
    let file_information = &editor.file_information;

    write!(screen, "{}", termion::clear::All).unwrap();
    let mut write_row = 3;
    for line in &file_information.contents
        [editor_status.display_begin_row..editor_status.display_end_row - 2]
    {
        write!(screen, "{}{}", termion::cursor::Goto(1, write_row), line).unwrap();
        write_row += 1;
    }
    write!(
        screen,
        "{}",
        termion::cursor::Goto(editor_status.cursor_col, editor_status.cursor_row)
    )
    .unwrap();
    screen.flush().unwrap();
}

fn handle_events(editor: &mut Editor, screen: &mut dyn Write) {
    let stdin = stdin();
    for c in stdin.events() {
        let evt = c.unwrap();
        match evt {
            Event::Key(Key::Ctrl('q')) => break,
            Event::Key(Key::Ctrl('s')) => {
                save_file("Hi.txt");
            }
            Event::Key(Key::Char(c)) => {
                if c as i32 == 10 {
                    editor.edit_status.cursor_row += 1;
                    editor.edit_status.cursor_col = 0;
                }
                editor.edit_status.cursor_col += 1;
                print!("{}", c)
            }
            Event::Key(Key::Left) => {
                if editor.edit_status.cursor_col == 0 {
                    continue;
                }
                editor.edit_status.cursor_col -= 1;
            }
            Event::Key(Key::Right) => {
                if editor.edit_status.cursor_col == editor.edit_status.width - 1 {
                    continue;
                }
                editor.edit_status.cursor_col += 1
            }
            Event::Key(Key::Up) => {
                if editor.edit_status.cursor_row == editor.edit_status.display_begin_row as u16 {
                    if editor.edit_status.cursor_row == 0 {
                        continue;
                    }
                    editor.edit_status.display_begin_row -= 1;
                    editor.edit_status.display_end_row -= 1;
                }
                editor.edit_status.cursor_row -= 1;
            }
            Event::Key(Key::Down) => {
                if editor.edit_status.cursor_row == editor.edit_status.display_end_row as u16 - 1 {
                    if editor.edit_status.cursor_row
                        == editor.file_information.contents.len() as u16 - 1
                    {
                        continue;
                    }
                    editor.edit_status.display_begin_row += 1;
                    editor.edit_status.display_end_row += 1;
                    continue;
                }
                editor.edit_status.cursor_row += 1;
            }
            Event::Key(Key::Backspace) => {
                if editor.edit_status.cursor_col == 0 {
                    continue;
                }
                editor.edit_status.cursor_col -= 1;
            }
            Event::Mouse(me) => match me {
                MouseEvent::Press(_, x, y) => {
                    write!(screen, "{}", termion::cursor::Goto(x, y)).unwrap();
                    screen.flush().unwrap();
                }
                _ => (),
            },
            _ => {}
        }
        display_file(editor, screen);
    }
}

fn save_file(path: &str) -> fs::File {
    let file = match fs::File::create(Path::new(path)) {
        Err(why) => panic!("couldn't create {}: {}", path, why),
        Ok(file) => file,
    };
    file
}

fn main() {
    let file_name = get_file_name();
    //TODO: Fix panic
    let (_, file_extension) = file_name.split_at(file_name.find('.').unwrap() + 1);
    let syntax_set = SyntaxSet::load_defaults_newlines();
    let syntax = syntax_set.find_syntax_by_extension(file_extension).unwrap();
    let theme = &ThemeSet::load_defaults().themes["base16-ocean.dark"];
    let terminal_size = terminal_size().unwrap();

    let mut editor = Editor {
        edit_configuration: EditorConfiguration {
            syntax: syntax,
            theme: theme,
        },
        edit_status: EditorStatus {
            width: terminal_size.0,
            height: terminal_size.1,
            display_begin_row: 0,
            display_end_row: terminal_size.1 as usize,
            cursor_row: 2,
            cursor_col: 1,
            saved: false,
        },
        file_information: FileInformation {
            file_path: PathBuf::from(file_name.as_str()),
            file_name: file_name,
            contents: Vec::new(),
        },
    };

    editor.load_file();
    let mut screen = create_editor_ui();
    display_file(&mut editor, &mut screen);
    handle_events(&mut editor, &mut screen);
}
