use crate::config::Lookup;
use log::{debug, error, info, warn};
use std::{collections::HashMap, process::Command};
use thiserror::Error;
#[derive(Error, Debug)]
pub enum PasswordStoreError {
    #[error("IO Error: '{0}'")]
    IOError(#[from] std::io::Error),
    #[error("Gpg returned error code '{0}'")]
    GpgCode(std::process::ExitStatus),
}

pub fn resolve_secrets(
    secrets: Vec<Lookup>,
) -> Result<HashMap<Lookup, Vec<u8>>, PasswordStoreError> {
    let mut map = HashMap::new();

    // does the IO to get the passwords
    for lookup in secrets.iter() {
        let mut command = Command::new("pass");
        command.arg(lookup.0.clone());
        debug!("{:#?}", command);
        let result = command.output()?;

        if result.status.code() != Some(0) {
            error!("{}", result.status);
            warn!("{}", String::from_utf8_lossy(&result.stderr));
            return Err(PasswordStoreError::GpgCode(result.status));
        } else {
            info!("{}", String::from_utf8_lossy(&result.stderr));
            let value = result.stdout;
            map.insert(lookup.clone(), value);
        }
    }
    Ok(map)
}
