mod config;
mod convert;
mod k8s;
mod passwordstore;

use std::fs::File;
use std::io::prelude::*;
use std::path::PathBuf;

use clap::Parser;
use config::Config;
use log::{debug, info};

const CONFIG_FILE: &str = "kubepassgen.toml";

/// Converts password store secrets into kubernetes yaml secrets
#[derive(Parser, Debug)]
#[clap(author, version, about, long_about = None)]
struct Args {
    /// Configuration file
    #[clap(short, long, parse(from_os_str), default_value = CONFIG_FILE)]
    config: PathBuf,

    /// Silence logging
    #[clap(short, long)]
    quiet: bool,

    /// Logging verbosity
    #[clap(short, long, parse(from_occurrences))]
    verbose: usize,

    /// Display timestamps
    #[clap(short, long, default_value = "none")]
    timestamp: stderrlog::Timestamp,
}

fn main() -> anyhow::Result<()> {
    let args = Args::parse();
    stderrlog::new()
        .module(module_path!())
        .quiet(args.quiet)
        .verbosity(args.verbose)
        .timestamp(args.timestamp)
        .init()?;

    let mut file = File::open(args.config)?;
    let mut contents = String::new();
    file.read_to_string(&mut contents)?;

    let config: Config = toml::from_str(&contents)?;
    debug!("Loaded config {:#?}", config);
    let to_resolve = config::list_keys(&config);
    info!("Looking for: {:#?}", to_resolve);
    let secrets = passwordstore::resolve_secrets(to_resolve)?;
    info!("Resolved secrets: {:#?}", secrets.keys());
    let kube = convert::opaque(&config.opaque, &secrets)?;
    debug!("Kube manifests {:#?}", kube);

    for manifest in kube {
        let text = serde_yaml::to_string(&manifest)?;
        println!("{}", text);
    }
    Ok(())
}
