use crate::config;
use crate::k8s;
use std::collections::BTreeMap;
use std::collections::HashMap;
use thiserror::Error;

#[derive(Error, Debug)]
pub enum ConvertError {
    #[error("Missing secret key: '{0:?}'")]
    MissingSecret(config::Lookup),
}
pub fn opaque(
    opaque: &HashMap<config::Name, config::Opaque>,
    secret_lookup: &HashMap<config::Lookup, Vec<u8>>,
) -> Result<Vec<k8s::Manifest>, ConvertError> {
    // Fins the secret in the lookup map. Failes if it doesn't exist
    let find_secret = |key: &config::Lookup| -> Result<Vec<u8>, ConvertError> {
        secret_lookup
            .get(key)
            .ok_or_else(|| ConvertError::MissingSecret(key.clone()))
            .cloned()
    };

    // Map the yaml tuple to the resolve k8s tuple
    let map_secret = | (key, lookup):(&config::Key, &config::Lookup) | -> Result<(k8s::Key,k8s::Base64), ConvertError> {

        let to_tuple = | bytes:Vec<u8>| -> (k8s::Key, k8s::Base64) {
            let key = k8s::Key(key.0.clone());
            let base64 = k8s::Base64::encode(&bytes);
            (key,base64)
        };
        find_secret(lookup).map(to_tuple)
    };

    // builds the k8s manifest in this namespace
    let manifest = |name: String, namespace: Option<String>, data: k8s::Data| -> k8s::Manifest {
        k8s::Manifest {
            api_version: k8s::ApiVersion::V1,
            kind: k8s::Kind::Secret,
            type_: k8s::Type::Opaque,
            metadata: k8s::Metadata { name, namespace },
            data: Some(data),
        }
    };

    // takes the high level opaque def and creates a file, or multiple if we are processing a namespace
    let to_files = |(name,value):(&config::Name,&config::Opaque)| -> Result<Vec<k8s::Manifest>, ConvertError> {

        let name = name.0.clone();
        match value {
            config::Opaque::Unspecified(keys) => {
                let mapped: Vec<(k8s::Key, k8s::Base64)> = keys.0.iter().map(map_secret).collect::<Result<_,_>>()?;
                let data: BTreeMap<k8s::Key, k8s::Base64> = mapped.into_iter().collect();
                let file = manifest(name,None,k8s::Data(data));
                Ok(vec![file])
            },
            config::Opaque::Namespaced(map) => {
                let namespace = Some(name);
                let files = map.iter().map(|(name,keys)|{
                    let name = name.0.clone();
                    let mapped: Vec<(k8s::Key, k8s::Base64)> = keys.0.iter().map(map_secret).collect::<Result<_,_>>()?;
                    let data: BTreeMap<k8s::Key, k8s::Base64> = mapped.into_iter().collect();
                    let file = manifest(name,namespace.clone(),k8s::Data(data));
                    Ok(file)
                });
                files.collect()
            },
        }
    };
    let results: Vec<Vec<_>> = opaque.iter().map(to_files).collect::<Result<_, _>>()?;
    Ok(results.into_iter().flatten().collect())
}
