use serde::Deserialize;
use std::collections::HashMap;

#[derive(Deserialize, Debug)]
pub struct Config {
    pub opaque: HashMap<Name, Opaque>,
}

#[derive(Deserialize, Debug)]
#[serde(untagged)]
pub enum Opaque {
    Unspecified(Keys),
    Namespaced(HashMap<Name, Keys>),
}

#[derive(Deserialize, Debug, Hash, PartialEq, Eq)]
#[serde(transparent)]
pub struct Name(pub String);

#[derive(Deserialize, Debug, Clone)]
pub struct Keys(pub HashMap<Key, Lookup>);

#[derive(Deserialize, Debug, Hash, PartialEq, Eq, Clone)]
#[serde(transparent)]
pub struct Key(pub String);

#[derive(Deserialize, Debug, Hash, PartialEq, Eq, Clone, PartialOrd, Ord)]
#[serde(transparent)]
pub struct Lookup(pub String);

/// Gets all the references in password store
pub fn list_keys(config: &Config) -> Vec<Lookup> {
    // get the lookups from an entry
    let get_lookup = |opaque: &Opaque| -> Vec<_> {
        // find the references we need to lookup
        let lookups =
            |keys: &Keys| -> Vec<Lookup> { keys.clone().0.into_iter().map(|(_, l)| l).collect() };

        match opaque {
            Opaque::Unspecified(keys) => lookups(keys),
            Opaque::Namespaced(map) => {
                let keys: Vec<Keys> = map.iter().map(|(_, k)| k).cloned().collect();
                let lookups = keys.iter().flat_map(lookups).collect();
                lookups
            }
        }
    };

    // a list of what to resolve in password store
    let mut lookups: Vec<Lookup> = config
        .opaque
        .iter()
        .flat_map(|(_, o)| get_lookup(o))
        .collect::<Vec<_>>();
    lookups.sort();
    lookups.dedup();

    lookups
}
