use crate::goodies::Time;
use chrono::{Datelike, Timelike, Utc};
use libmacchina::traits::ReadoutError;
use libmacchina::BatteryReadout;
use libmacchina::GeneralReadout;
use libmacchina::MemoryReadout;

pub fn uptime() -> Result<Time, ReadoutError> {
    use libmacchina::traits::GeneralReadout as _;

    let read = GeneralReadout::new();
    if let Ok(up) = read.uptime() {
        return Ok(Time::from(up));
    }

    Err(ReadoutError::Warning(
        "I couldn't get this statistic :(".to_string(),
    ))
}

pub fn network(interface: Option<String>) -> Result<String, ReadoutError> {
    use libmacchina::traits::GeneralReadout as _;

    let read = GeneralReadout::new();

    match read.local_ip(interface) {
        Ok(ip) => Ok(format!("Local IP: {}\nState: Online", ip)),
        Err(e) => Err(ReadoutError::Warning(e.to_string())),
    }
}

pub fn memory() -> Result<String, ReadoutError> {
    use bytesize::ByteSize;
    use libmacchina::traits::MemoryReadout as _;

    let read = MemoryReadout::new();

    match (read.used(), read.total()) {
        (Ok(u), Ok(t)) => {
            let total = ByteSize::kb(t);
            let used = ByteSize::kb(u);
            let percentage = u as f64 / t as f64 * 100f64;

            Ok(format!("{}/{} ({:.1}%)", used, total, percentage))
        }
        _ => Err(ReadoutError::Warning(
            "I couldn't get this statistic :(".to_string(),
        )),
    }
}

pub fn battery() -> Result<String, ReadoutError> {
    use libmacchina::traits::BatteryReadout as _;

    let read = BatteryReadout::new();
    match (read.percentage(), read.status()) {
        (Ok(p), Ok(s)) => match s {
            libmacchina::traits::BatteryState::Charging => {
                Ok(format!("Percentage: {}%\nState: Charging", p))
            }
            libmacchina::traits::BatteryState::Discharging => {
                Ok(format!("Percentage: {}%\nState: Discharging", p))
            }
        },
        _ => Err(ReadoutError::Warning(
            "I couldn't get this statistic :(".to_string(),
        )),
    }
}

pub fn date_time() -> Result<String, ReadoutError> {
    let now = Utc::now();
    let (is_pm, hour) = now.hour12();
    let date = format!("{}-{:02}-{:02}", now.year(), now.month(), now.day());
    let time = format!(
        "{:02}:{:02}:{:02} {}",
        hour,
        now.minute(),
        now.second(),
        if is_pm { "PM" } else { "AM" }
    );

    Ok(format!("{} at {}", date, time))
}
