use std::path::PathBuf;
use std::time::Instant;

use argh::FromArgs;
use shtola::log::*;
use shtola::plugins;
use shtola::Shtola;

const VERSION: &'static str = "0.5.0";

#[derive(FromArgs)]
/// A static site generator.
struct Args {
	/// output the current version
	#[argh(switch)]
	version: bool,

	/// the source directory
	#[argh(option, short = 's', default = "PathBuf::from(\"./\")")]
	source: PathBuf,

	/// the destination directory
	#[argh(option, short = 'd', default = "PathBuf::from(\"./dest\")")]
	destination: PathBuf,

	/// whether to wipe the destination directory before building
	#[argh(switch, short = 'c')]
	clean: bool,

	/// what plugins to not run
	#[argh(option, short = 'i')]
	ignored_plugins: Vec<String>,

	/// whether to skip parsing frontmatter (will probably break plugins)
	#[argh(switch)]
	no_frontmatter: bool,

	/// whether to output debug stuff
	#[argh(switch, short = 'v')]
	verbose: bool,
}

fn main() {
	let args: Args = argh::from_env();

	if args.version {
		println!("{}", VERSION);
		std::process::exit(0);
	}

	if args.verbose {
		minifemme::start(minifemme::LevelFilter::Trace, minifemme::LogMode::Pretty);
	} else {
		minifemme::start(minifemme::LevelFilter::Info, minifemme::LogMode::Pretty);
	}

	info!("Starting build");
	let start = Instant::now();
	let mut sh = Shtola::new();
	sh.source(&args.source);
	sh.destination(args.destination);
	sh.clean(args.clean);
	sh.frontmatter(!args.no_frontmatter);
	let potential_gitignore = args.source.join(".gitignore");
	if potential_gitignore.exists() {
		sh.source_ignores(&potential_gitignore).unwrap();
	}
	let potential_krileignore = args.source.join(".krileignore");
	if potential_krileignore.exists() {
		sh.source_ignores(&potential_krileignore).unwrap();
	}
	if !args.ignored_plugins.contains(&"tera_layouts".to_string()) {
		sh.register(plugins::tera_layouts::plugin());
	}
	if !args.ignored_plugins.contains(&"markdown".to_string()) {
		sh.register(plugins::markdown::plugin());
	}
	sh.build().unwrap();
	info!(
		"Finished build in {:?}",
		Instant::now().duration_since(start)
	);
}
