use std::path::PathBuf;
use std::time::Instant;

use argh::FromArgs;
use shtola::log::*;
use shtola::plugins;
use shtola::Shtola;

#[derive(FromArgs)]
/// A static site generator.
struct Args {
	/// the source directory
	#[argh(positional)]
	source: PathBuf,

	/// the destination directory
	#[argh(positional, default = "PathBuf::from(\"./dest\")")]
	destination: PathBuf,

	/// whether to wipe the destination directory before building
	#[argh(switch, short = 'c')]
	clean: bool,

	/// whether to skip parsing frontmatter (will probably break plugins)
	#[argh(switch, short = 'f')]
	frontmatter: bool,

	/// what plugins to not run
	#[argh(option, short = 'i')]
	ignored_plugins: Vec<String>,
}

fn main() {
	minifemme::start(minifemme::LevelFilter::Info, minifemme::LogMode::Pretty);
	info!("Starting build");
	let start = Instant::now();
	let args: Args = argh::from_env();
	let mut sh = Shtola::new();
	sh.source(args.source);
	sh.destination(args.destination);
	sh.clean(args.clean);
	sh.frontmatter(!args.frontmatter);
	if !args.ignored_plugins.contains(&"tera_layouts".to_string()) {
		sh.register(plugins::tera_layouts::plugin());
	}
	if !args.ignored_plugins.contains(&"markdown".to_string()) {
		sh.register(plugins::markdown::plugin());
	}
	sh.build().unwrap();
	info!(
		"Finished build in {:?}",
		Instant::now().duration_since(start)
	);
}
