//! Definitions of Koopa IR tokens.
//!
//! Tokens can represent integer literals, symbols, keywords, binary
//! operators, characters and EOFs. The Koopa IR lexer
//! ([`Lexer`](crate::front::lexer::Lexer)) will produce tokens during
//! the lexing process.

use crate::front::span::Span;
use crate::ir::BinaryOp;
use std::fmt;

/// Tokens that will be generated by the lexer.
pub struct Token {
  pub span: Span,
  pub kind: TokenKind,
}

impl Token {
  /// Creates a new token.
  pub fn new(span: Span, kind: TokenKind) -> Self {
    Self { span, kind }
  }
}

impl Default for Token {
  /// Creates a new token with default span and `End` kind.
  fn default() -> Self {
    Self {
      span: Span::default(),
      kind: TokenKind::End,
    }
  }
}

/// Kind of token.
#[derive(Debug, PartialEq)]
pub enum TokenKind {
  /// Integer literal.
  Int(i64),
  /// Symbol (identifier like `@id` or `%id`).
  Symbol(String),
  /// Keyword.
  Keyword(Keyword),
  /// Binary operator.
  BinaryOp(BinaryOp),
  /// Other characters.
  Other(char),
  /// End of file.
  End,
}

impl fmt::Display for TokenKind {
  fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
    match self {
      TokenKind::Int(v) => write!(f, "integer '{}'", v),
      TokenKind::Symbol(v) => write!(f, "symbol '{}'", v),
      TokenKind::Keyword(v) => write!(f, "keyword '{}'", v),
      TokenKind::BinaryOp(v) => write!(f, "binary operator '{}'", v),
      TokenKind::Other(v) => write!(f, "character '{}'", v),
      TokenKind::End => write!(f, "end of file"),
    }
  }
}

/// Keywords of Koopa IR.
#[derive(Clone, Copy, Debug, PartialEq)]
pub enum Keyword {
  /// Keyword `i32`.
  I32,
  /// Keyword `undef`.
  Undef,
  /// Keyword `zeroinit`.
  ZeroInit,
  /// Keyword `global`.
  Global,
  /// Keyword `alloc`.
  Alloc,
  /// Keyword `load`.
  Load,
  /// Keyword `store`.
  Store,
  /// Keyword `getptr`.
  GetPtr,
  /// Keyword `getelemptr`.
  GetElemPtr,
  /// Keyword `br`.
  Br,
  /// Keyword `jump`.
  Jump,
  /// Keyword `call`.
  Call,
  /// Keyword `ret`.
  Ret,
  /// Keyword `fun`.
  Fun,
  /// Keyword `decl`.
  Decl,
}

impl fmt::Display for Keyword {
  fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
    match self {
      Keyword::I32 => f.write_str("i32"),
      Keyword::Undef => f.write_str("undef"),
      Keyword::ZeroInit => f.write_str("zeroinit"),
      Keyword::Global => f.write_str("global"),
      Keyword::Alloc => f.write_str("alloc"),
      Keyword::Load => f.write_str("load"),
      Keyword::Store => f.write_str("store"),
      Keyword::GetPtr => f.write_str("getptr"),
      Keyword::GetElemPtr => f.write_str("getelemptr"),
      Keyword::Br => f.write_str("br"),
      Keyword::Jump => f.write_str("jump"),
      Keyword::Call => f.write_str("call"),
      Keyword::Ret => f.write_str("ret"),
      Keyword::Fun => f.write_str("fun"),
      Keyword::Decl => f.write_str("decl"),
    }
  }
}
