use crate::FType;
use crate::Vec3;

// Spectral sensitivity curve of CIE tristumulus X.
// Sensitivity of electromagnetic radiation at certain wavelengths, mapping to CIE XYZ
// Ranges from 375..785nm at a step of 5nm
// 0.0 has been prepended to the dataset for simpler interpolation code
const X_STIMULUS: [FType; 83] = [
    0.0, 0.001368, 0.002236, 0.004243, 0.007650, 0.014310, 0.023190, 0.043510, 0.077630, 0.134380,
    0.214770, 0.283900, 0.328500, 0.348280, 0.348060, 0.336200, 0.318700, 0.290800, 0.251100,
    0.195360, 0.142100, 0.095640, 0.057950, 0.032010, 0.014700, 0.004900, 0.002400, 0.009300,
    0.029100, 0.063270, 0.109600, 0.165500, 0.225750, 0.290400, 0.359700, 0.433450, 0.512050,
    0.594500, 0.678400, 0.762100, 0.842500, 0.916300, 0.978600, 1.026300, 1.056700, 1.062200,
    1.045600, 1.002600, 0.938400, 0.854450, 0.751400, 0.642400, 0.541900, 0.447900, 0.360800,
    0.283500, 0.218700, 0.164900, 0.121200, 0.087400, 0.063600, 0.046770, 0.032900, 0.022700,
    0.015840, 0.011359, 0.008111, 0.005790, 0.004109, 0.002899, 0.002049, 0.001440, 0.001000,
    0.000690, 0.000476, 0.000332, 0.000235, 0.000166, 0.000117, 0.000083, 0.000059, 0.000042, 0.0,
];

// Spectral sensitivity curve of CIE tristumulus Y.
// Sensitivity of electromagnetic radiation at certain wavelengths, mapping to CIE XYZ
// Ranges from 375..785nm at a step of 5nm
// 0.0 has been prepended and appended to the dataset for simpler interpolation code
const Y_STIMULUS: [FType; 83] = [
    0.0, 0.000039, 0.000064, 0.000120, 0.000217, 0.000396, 0.000640, 0.001210, 0.002180, 0.004000,
    0.007300, 0.011600, 0.016840, 0.023000, 0.029800, 0.038000, 0.048000, 0.060000, 0.073900,
    0.090980, 0.112600, 0.139020, 0.169300, 0.208020, 0.258600, 0.323000, 0.407300, 0.503000,
    0.608200, 0.710000, 0.793200, 0.862000, 0.914850, 0.954000, 0.980300, 0.994950, 1.000000,
    0.995000, 0.978600, 0.952000, 0.915400, 0.870000, 0.816300, 0.757000, 0.694900, 0.631000,
    0.566800, 0.503000, 0.441200, 0.381000, 0.321000, 0.265000, 0.217000, 0.175000, 0.138200,
    0.107000, 0.081600, 0.061000, 0.044580, 0.032000, 0.023200, 0.017000, 0.011920, 0.008210,
    0.005723, 0.004102, 0.002929, 0.002091, 0.001484, 0.001047, 0.000740, 0.000520, 0.000361,
    0.000249, 0.000172, 0.000120, 0.000085, 0.000060, 0.000042, 0.000030, 0.000021, 0.000015, 0.0,
];

// Spectral sensitivity curve of CIE tristumulus Z.
// Sensitivity of electromagnetic radiation at certain wavelengths, mapping to CIE XYZ
// Ranges from 375..785nm at a step of 5nm
// 0.0 has been prepended to the dataset for simpler interpolation code
const Z_STIMULUS: [FType; 83] = [
    0.0, 0.006450, 0.010550, 0.020050, 0.036210, 0.067850, 0.110200, 0.207400, 0.371300, 0.645600,
    1.039050, 1.385600, 1.622960, 1.747060, 1.782600, 1.772110, 1.744100, 1.669200, 1.528100,
    1.287640, 1.041900, 0.812950, 0.616200, 0.465180, 0.353300, 0.272000, 0.212300, 0.158200,
    0.111700, 0.078250, 0.057250, 0.042160, 0.029840, 0.020300, 0.013400, 0.008750, 0.005750,
    0.003900, 0.002750, 0.002100, 0.001800, 0.001650, 0.001400, 0.001100, 0.001000, 0.000800,
    0.000600, 0.000340, 0.000240, 0.000190, 0.000100, 0.000050, 0.000030, 0.000020, 0.000010,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000,
    0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.000000, 0.0,
];

fn gaussian(x: FType, alpha: FType, μ: FType, σ1: FType, σ2: FType) -> FType {
    let t = (x - μ) / if x < μ { σ1 } else { σ2 };
    alpha * (-(t * t) / 2.0).exp()
}

/// Returns the spectral sensitivity yielding XYZ tristumulus for a given electromagnetic wavelength
/// specified in nanometers, using an analytical approximation.
/// Source: http://jcgt.org/published/0002/02/01/
#[allow(non_snake_case)]
pub fn XYZ_sensitivity_from_wavelength_approx(λ: FType) -> Vec3 {
    let λ = λ * 10.0; // convert to ångströms, the unit of the input to the analytical approximation
    let x = gaussian(λ, 1.056, 5998.0, 379.0, 310.0)
        + gaussian(λ, 0.362, 4420.0, 160.0, 267.0)
        + gaussian(λ, -0.065, 5011.0, 204.0, 262.0);
    let y = gaussian(λ, 0.821, 5688.0, 469.0, 405.0) + gaussian(λ, 0.286, 5309.0, 163.0, 311.0);
    let z = gaussian(λ, 1.217, 4370.0, 118.0, 360.0) + gaussian(λ, 0.681, 4590.0, 260.0, 138.0);
    Vec3::new(x, y, z)
}

/// Returns the spectral sensitivity yielding XYZ tristumulus for a given electromagnetic wavelength
/// specified in nanometers, using the CIE 1931 2° standard observer data table.
#[allow(non_snake_case)]
pub fn XYZ_sensitivity_from_wavelength(λ: FType) -> Vec3 {
    let bucket_idx = (λ - 375.0) / 5.0;
    let bucket_1 = bucket_idx.max(0.0);
    let bucket_1_idx = (bucket_1 as usize).min(X_STIMULUS.len() - 1);
    let sample_1 = Vec3::new(
        X_STIMULUS[bucket_1_idx],
        Y_STIMULUS[bucket_1_idx],
        Z_STIMULUS[bucket_1_idx],
    );

    let bucket_2_idx = (bucket_1 as usize + 1).min(X_STIMULUS.len() - 1);
    let t = bucket_idx.fract();
    if t == 0.0 {
        return sample_1;
    }
    let sample_2 = Vec3::new(
        X_STIMULUS[bucket_2_idx],
        Y_STIMULUS[bucket_2_idx],
        Z_STIMULUS[bucket_2_idx],
    );

    sample_1.lerp(sample_2, t)
}

#[allow(non_snake_case)]
pub fn XYZ_from_spd(spd: &[FType], wavelength_start: FType, wavelength_end: FType) -> Vec3 {
    let step = (wavelength_end - wavelength_start) / spd.len() as FType;
    let mut XYZ = Vec3::ZERO;
    for (idx, power) in spd.iter().enumerate() {
        let wavelength = wavelength_start + step * idx as FType;
        XYZ += XYZ_sensitivity_from_wavelength(wavelength) * *power * step;
    }
    XYZ
}

#[cfg(test)]
mod test {
    use crate::{ColorConversion, ColorSpace};

    use super::*;

    fn parse_spd_file(path: &str) -> (String, Vec<FType>) {
        let spd_file = std::fs::read_to_string(path).unwrap();
        let mut lines = spd_file.split("\n");
        let header = lines.next().unwrap();
        let mut spd = Vec::new();
        for line in lines {
            let trimmed = line.trim();
            if trimmed == "" {
                break;
            }
            let mut power = trimmed.parse::<FType>().unwrap();
            if power == -1.2300000e+034 {
                power = 0.0;
            }
            spd.push(power);
        }
        (header.to_string(), spd)
    }
    const D65_SPD: [FType; 400] = [
        0.49975500, 0.50442800, 0.50910000, 0.51377300, 0.51844600, 0.52311800, 0.52779100,
        0.53246400, 0.53713700, 0.54180900, 0.54648200, 0.57458900, 0.60269500, 0.63080200,
        0.65890900, 0.68701500, 0.71512200, 0.74322900, 0.77133600, 0.79944200, 0.82754900,
        0.83628000, 0.84501100, 0.85374200, 0.86247300, 0.87120400, 0.87993600, 0.88866700,
        0.89739800, 0.90612900, 0.91486000, 0.91680600, 0.91875200, 0.92069700, 0.92264300,
        0.92458900, 0.92653500, 0.92848100, 0.93042600, 0.93237200, 0.93431800, 0.92756800,
        0.92081900, 0.91406900, 0.90732000, 0.90057000, 0.89382100, 0.88707100, 0.88032200,
        0.87357200, 0.86682300, 0.88500600, 0.90318800, 0.92137100, 0.93955400, 0.95773600,
        0.97591900, 0.99410200, 1.01228000, 1.03047000, 1.04865000, 1.06079000, 1.07294000,
        1.08508000, 1.09722000, 1.10936000, 1.12151000, 1.13365000, 1.14579000, 1.15794000,
        1.17008000, 1.17088000, 1.17169000, 1.17249000, 1.17330000, 1.17410000, 1.17490000,
        1.17571000, 1.17651000, 1.17732000, 1.17812000, 1.17517000, 1.17222000, 1.16927000,
        1.16632000, 1.16336000, 1.16041000, 1.15746000, 1.15451000, 1.15156000, 1.14861000,
        1.14967000, 1.15073000, 1.15180000, 1.15286000, 1.15392000, 1.15498000, 1.15604000,
        1.15711000, 1.15817000, 1.15923000, 1.15212000, 1.14501000, 1.13789000, 1.13078000,
        1.12367000, 1.11656000, 1.10945000, 1.10233000, 1.09522000, 1.08811000, 1.08865000,
        1.08920000, 1.08974000, 1.09028000, 1.09082000, 1.09137000, 1.09191000, 1.09245000,
        1.09300000, 1.09354000, 1.09199000, 1.09044000, 1.08888000, 1.08733000, 1.08578000,
        1.08423000, 1.08268000, 1.08112000, 1.07957000, 1.07802000, 1.07501000, 1.07200000,
        1.06898000, 1.06597000, 1.06296000, 1.05995000, 1.05694000, 1.05392000, 1.05091000,
        1.04790000, 1.05080000, 1.05370000, 1.05660000, 1.05950000, 1.06239000, 1.06529000,
        1.06819000, 1.07109000, 1.07399000, 1.07689000, 1.07361000, 1.07032000, 1.06704000,
        1.06375000, 1.06047000, 1.05719000, 1.05390000, 1.05062000, 1.04733000, 1.04405000,
        1.04369000, 1.04333000, 1.04297000, 1.04261000, 1.04225000, 1.04190000, 1.04154000,
        1.04118000, 1.04082000, 1.04046000, 1.03641000, 1.03237000, 1.02832000, 1.02428000,
        1.02023000, 1.01618000, 1.01214000, 1.00809000, 1.00405000, 1.00000000, 0.99633400,
        0.99266800, 0.98900300, 0.98533700, 0.98167100, 0.97800500, 0.97433900, 0.97067400,
        0.96700800, 0.96334200, 0.96279600, 0.96225000, 0.96170300, 0.96115700, 0.96061100,
        0.96006500, 0.95951900, 0.95897200, 0.95842600, 0.95788000, 0.95077800, 0.94367500,
        0.93657300, 0.92947000, 0.92236800, 0.91526600, 0.90816300, 0.90106100, 0.89395800,
        0.88685600, 0.88817700, 0.88949700, 0.89081800, 0.89213800, 0.89345900, 0.89478000,
        0.89610000, 0.89742100, 0.89874100, 0.90006200, 0.89965500, 0.89924800, 0.89884100,
        0.89843400, 0.89802600, 0.89761900, 0.89721200, 0.89680500, 0.89639800, 0.89599100,
        0.89409100, 0.89219000, 0.89029000, 0.88838900, 0.88648900, 0.88458900, 0.88268800,
        0.88078800, 0.87888700, 0.87698700, 0.87257700, 0.86816700, 0.86375700, 0.85934700,
        0.85493600, 0.85052600, 0.84611600, 0.84170600, 0.83729600, 0.83288600, 0.83329700,
        0.83370700, 0.83411800, 0.83452800, 0.83493900, 0.83535000, 0.83576000, 0.83617100,
        0.83658100, 0.83699200, 0.83332000, 0.82964700, 0.82597500, 0.82230200, 0.81863000,
        0.81495800, 0.81128500, 0.80761300, 0.80394000, 0.80026800, 0.80045600, 0.80064400,
        0.80083100, 0.80101900, 0.80120700, 0.80139500, 0.80158300, 0.80177000, 0.80195800,
        0.80214600, 0.80420900, 0.80627200, 0.80833600, 0.81039900, 0.81246200, 0.81452500,
        0.81658800, 0.81865200, 0.82071500, 0.82277800, 0.81878400, 0.81479100, 0.81079700,
        0.80680400, 0.80281000, 0.79881600, 0.79482300, 0.79082900, 0.78683600, 0.78284200,
        0.77427900, 0.76571600, 0.75715300, 0.74859000, 0.74002700, 0.73146500, 0.72290200,
        0.71433900, 0.70577600, 0.69721300, 0.69910100, 0.70098900, 0.70287600, 0.70476400,
        0.70665200, 0.70854000, 0.71042800, 0.71231500, 0.71420300, 0.71609100, 0.71883100,
        0.72157100, 0.72431100, 0.72705100, 0.72979000, 0.73253000, 0.73527000, 0.73801000,
        0.74075000, 0.74349000, 0.73074500, 0.71800000, 0.70525500, 0.69251000, 0.67976500,
        0.66702000, 0.65427500, 0.64153000, 0.62878500, 0.61604000, 0.62432200, 0.63260300,
        0.64088500, 0.64916600, 0.65744800, 0.66573000, 0.67401100, 0.68229300, 0.69057400,
        0.69885600, 0.70405700, 0.70925900, 0.71446000, 0.71966200, 0.72486300, 0.73006400,
        0.73526600, 0.74046700, 0.74566900, 0.75087000, 0.73937600, 0.72788100, 0.71638700,
        0.70489300, 0.69339800, 0.68190400, 0.67041000, 0.65891600, 0.64742100, 0.63592700,
        0.61875200, 0.60157800, 0.58440300, 0.56722900, 0.55005400, 0.53288000, 0.51570500,
        0.49853100, 0.48135600, 0.46418200, 0.48456900, 0.50495600, 0.52534400, 0.54573100,
        0.56611800, 0.58650500, 0.60689200, 0.62728000, 0.64766700, 0.66805400, 0.66463100,
        0.66120900, 0.65778600, 0.65436400, 0.65094100, 0.64751800, 0.64409600, 0.64067300,
        0.63725100,
    ];
    #[test]
    #[allow(non_snake_case)]
    fn test_XYZ_from_spd() {
        let (_, spd) = parse_spd_file("splib07a_LawnGrass_GDS91b_+const_1.0_BECKa_AREF.txt");
        let (_, wavelengths) =
            parse_spd_file("splib07a_Wavelengths_BECK_Beckman_0.2-3.0_microns.txt");
        let mut XYZ = Vec3::ZERO;
        let N = XYZ_from_spd(&D65_SPD, 380.0, 780.0);
        for (idx, power) in spd.iter().enumerate() {
            let wavelength = wavelengths[idx] * 1000.0;
            if wavelength < 380.0 || wavelength > 780.0 {
                continue;
            }
            let step = wavelengths[idx + 1] * 1000.0 - wavelength;
            let sensitivity = XYZ_sensitivity_from_wavelength(wavelength);
            let illuminant_idx =
                ((wavelength - 380.0).max(0.0) as usize).clamp(0, D65_SPD.len() - 1);
            let illuminant = D65_SPD[illuminant_idx] / step;
            XYZ += *power * sensitivity * illuminant * step;
            println!(
                "{}: i {} * s {} * r {} = {}",
                wavelength,
                illuminant,
                sensitivity,
                *power,
                illuminant * sensitivity * *power
            );
        }
        XYZ *= 1.0 / N.y;
        let conversion = ColorConversion::new(crate::spaces::CIE_XYZ, crate::spaces::LINEAR_SRGB);
        let srgb = conversion.convert(XYZ);
        println!("XYZ: {:?} srgb: {:?} N: {:?}", XYZ, srgb, N);
    }

    #[test]
    #[allow(non_snake_case)]
    fn test_XYZ_sensitivity_error() {
        let step = 0.1;
        for idx in 0..((780.0 - 380.0) / step) as usize {
            let wavelength = 380.0 + idx as FType * step;
            let table = XYZ_sensitivity_from_wavelength(wavelength);
            let approx = XYZ_sensitivity_from_wavelength_approx(wavelength);
            // check error between the approximation and the table source-of-truth
            let error = (approx - table).abs();
            assert!(
                error.x < 0.05,
                "error.x was {} for wavelength {}",
                error.x,
                wavelength
            );
            assert!(
                error.y < 0.05,
                "error.y was {} for wavelength {}",
                error.y,
                wavelength
            );
            assert!(
                error.z < 0.05,
                "error.z was {} for wavelength {}",
                error.z,
                wavelength
            );
        }
    }
}
