use std::collections::HashMap;
use std::str::FromStr;
use std::io::{self, Write};
use std::fmt;

use crate::error::{Error::{self, *}, Result};
use crate::options::ParseArgs;

#[derive(Debug)]
pub struct Data {
    schemas: HashMap<String, TableSchema>,
    entries: Vec<DataEntry>,
}

impl Data {
    pub fn new(schemas: HashMap<String, TableSchema>, entries: Vec<DataEntry>) -> Data {
        Data {
            schemas, entries,
        }
    }

    pub fn add_schema(&mut self, schema: TableSchema) {
        self.schemas.insert(schema.name().into(), schema);
    }

    pub fn add_entry(&mut self, entry: DataEntry) {
        self.entries.push(entry);
    }

    pub fn render<W: Write>(self, mut device: W, args: ParseArgs) -> io::Result<()> {
        for schema in self.schemas.values() {
            device.write_all(format!("\
CREATE TABLE {table_name} (
    {id_col} INTEGER NOT NULL,
    {col_definitions},
    {geom_field} geometry({geom_type}, {srs}) NOT NULL
);

CREATE SEQUENCE {table_name}_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE {table_name}_id_seq OWNED BY {table_name}.{id_col};

ALTER TABLE ONLY {table_name} ALTER COLUMN {id_col} SET DEFAULT nextval('{table_name}_id_seq'::regclass);

",
                table_name = schema.name().to_owned(),
                id_col = args.id_col,
                geom_field = args.geom_field,
                geom_type = "GEOMETRYCOLLECTION",
                srs = 4326,
                col_definitions = schema.column_definitions(args.id_col),
            ).as_bytes())?;
        }

        let mut sorted_keys = HashMap::new();
        let mut max_ids = HashMap::new();

        for entry in self.entries {
            let schema = entry.schema().unwrap();
            let keys = sorted_keys.entry(schema.clone()).or_insert_with(|| {
                entry.sorted_keys()
            });

            let entry_id = entry.id(args.id_col);
            let max_id = max_ids.entry(schema.clone()).or_insert_with(|| {
                entry_id
            });

            if entry.id(args.id_col) > *max_id {
                *max_id = entry_id;
            }

            device.write(format!("\
                INSERT INTO {table_name} ({column_definitions}, {geom_field}) VALUES ({values}, ST_GeomFromText('{geometry}'));\n",
                table_name = entry.schema().unwrap(),
                column_definitions = keys.join(","),
                values = entry.values(keys),
                geom_field = args.geom_field,
                geometry = entry.geometry().unwrap_or_else(|| "NULL".to_owned()),
            ).as_bytes())?;
        }

        device.write(b"\n\n")?;

        for (key, value) in max_ids {
            device.write(format!(
                "SELECT pg_catalog.setval('{table_name}_id_seq', {value}, true);\n",
                table_name = key,
                value = value,
            ).as_bytes())?;
        }

        Ok(())
    }
}

#[derive(Debug, Copy, Clone)]
pub enum ColType {
    String,
    Int,
    Float,
}

impl ColType {
    fn validate(&self, data: String) -> Result<ColValue> {
        match self {
            &ColType::Int => Ok(ColValue::Int(data.parse().unwrap())),
            &ColType::Float => Ok(ColValue::Float(data.parse().unwrap())),
            &ColType::String => Ok(ColValue::String(data)),
        }
    }
}

impl fmt::Display for ColType {
    fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        write!(formatter, "{}", match self {
            ColType::String => "VARCHAR",
            ColType::Int => "INTEGER",
            ColType::Float => "DOUBLE PRECISION",
        })
    }
}

#[derive(Debug)]
pub enum ColValue {
    String(String),
    Int(i64),
    Float(f64),
}

impl FromStr for ColType {
    type Err = Error;

    fn from_str(s: &str) -> Result<ColType> {
        Ok(match s {
            "string" => ColType::String,
            "int" => ColType::Int,
            "float" => ColType::Float,
            x => {
                return Err(UnknownColumnType(x.into()));
            }
        })
    }
}

#[derive(Debug)]
pub struct TableSchema {
    name: String,
    fields: HashMap<String, ColType>,
}

impl TableSchema {
    pub fn with_name(name: String) -> TableSchema {
        TableSchema {
            name,
            fields: HashMap::new(),
        }
    }

    pub fn name(&self) -> &str {
        &self.name
    }

    pub fn add_field(&mut self, name: String, r#type: ColType) {
        self.fields.insert(name, r#type);
    }

    pub fn validate_data(&self, field: &str, data: String) -> Result<ColValue> {
        self.fields.get(field).unwrap().validate(data)
    }

    pub fn column_definitions(&self, id_col: &str) -> String {
        self.fields.iter().filter(|(k, _v)| *k != id_col).map(|(k, v)| {
            format!("{} {}", k, v)
        }).collect::<Vec<_>>().join(",")
    }
}

#[derive(Debug)]
pub struct DataEntry {
    schema: Option<String>,
    data_fields: HashMap<String, ColValue>,
    geom: Option<Box<dyn Geometry>>,
}

impl DataEntry {
    pub fn new() -> DataEntry {
        DataEntry {
            schema: None,
            data_fields: HashMap::new(),
            geom: None,
        }
    }

    pub fn set_schema(&mut self, schema: String) {
        self.schema = Some(schema);
    }

    pub fn set_geom(&mut self, geom: Box<dyn Geometry>) {
        self.geom = Some(geom);
    }

    pub fn schema(&self) -> Option<String> {
        self.schema.clone()
    }

    pub fn add_data(&mut self, name: String, value: ColValue) {
        self.data_fields.insert(name, value);
    }

    pub fn id(&self, id_col: &str) -> i64 {
        if let Some(&ColValue::Int(i)) = self.data_fields.get(id_col) {
            i
        } else {
            panic!()
        }
    }

    pub fn geometry(&self) -> Option<String> {
        self.geom.as_ref().map(|g| g.to_string())
    }

    pub fn sorted_keys(&self) -> Vec<String> {
        let mut keys: Vec<_> = self.data_fields.keys().map(|k| k.clone()).collect();

        keys.sort_unstable();

        keys
    }

    pub fn values(&self, keys: &[String]) -> String {
        keys.iter().map(|k| {
            match self.data_fields.get(k) {
                Some(&ColValue::Int(i)) => i.to_string(),
                Some(&ColValue::Float(f)) => f.to_string(),
                Some(&ColValue::String(ref s)) => format!("'{}'", s.replace('\'', r"''")),
                None => "NULL".to_owned(),
            }
        }).collect::<Vec<_>>().join(", ")
    }
}

#[derive(Debug)]
pub struct EntryAttr {
    name: String,
    schema_name: String,
    value: Option<ColValue>,
}

impl EntryAttr {
    pub fn with_name_and_schema(name: String, schema_name: String) -> EntryAttr {
        EntryAttr {
            name, schema_name,
            value: None,
        }
    }

    pub fn name(&self) -> &str {
        &self.name
    }

    pub fn schema_name(&self) -> &str {
        &self.schema_name
    }

    pub fn set_value(&mut self, value: ColValue) {
        self.value = Some(value);
    }

    pub fn value(self) -> Option<ColValue> {
        self.value
    }
}

#[derive(Debug)]
pub struct Coordinates {
    coordinates: Vec<(f64, f64)>,
}

impl Coordinates {
    pub fn new() -> Coordinates {
        Coordinates {
            coordinates: Vec::new(),
        }
    }

    pub fn set_coordinates(&mut self, coords: Vec<(f64, f64)>) {
        self.coordinates = coords;
    }
}

impl fmt::Display for Coordinates {
    fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        write!(
            formatter,
            "({coords})",
            coords = self.coordinates.iter().map(|&(x, y)| format!("{} {}", x, y)).collect::<Vec<_>>().join(", ")
        )
    }
}

pub trait Geometry: fmt::Debug + fmt::Display { }

#[derive(Debug)]
pub struct GeometryCollection {
    geometries: Vec<Box<dyn Geometry>>,
}

impl GeometryCollection {
    pub fn new() -> GeometryCollection {
        GeometryCollection {
            geometries: Vec::new(),
        }
    }

    pub fn add_geometry(&mut self, geom: Box<dyn Geometry>) {
        self.geometries.push(geom);
    }
}

impl fmt::Display for GeometryCollection {
    fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        write!(
            formatter,
            "GEOMETRYCOLLECTION({geometries})",
            geometries = self.geometries.iter().map(|g| g.to_string()).collect::<Vec<_>>().join(", "),
        )
    }
}

impl Geometry for GeometryCollection { }

#[derive(Debug)]
pub struct Boundary {
    ring: Option<Coordinates>,
}

impl Boundary {
    pub fn new() -> Boundary {
        Boundary {
            ring: None,
        }
    }

    pub fn set_linear_ring(&mut self, lr: LinearRing) {
        self.ring = Some(lr.coordinates.unwrap());
    }
}

#[derive(Debug)]
pub struct Polygon {
    outerboundary: Option<Coordinates>,
    innerboundaries: Vec<Coordinates>,
}

impl Polygon {
    pub fn new() -> Polygon {
        Polygon {
            outerboundary: None,
            innerboundaries: Vec::new(),
        }
    }

    pub fn set_outer_boundary(&mut self, ob: Boundary) {
        self.outerboundary = Some(ob.ring.unwrap());
    }

    pub fn add_inner_boundary(&mut self, ib: Boundary) {
        self.innerboundaries.push(ib.ring.unwrap());
    }
}

impl fmt::Display for Polygon {
    fn fmt(&self, formatter: &mut fmt::Formatter) -> fmt::Result {
        let inner = if self.innerboundaries.is_empty() {
            "".to_owned()
        } else {
            format!(", {}", self.innerboundaries.iter().map(|ib| ib.to_string()).collect::<Vec<_>>().join(", "))
        };

        write!(
            formatter,
            "POLYGON ({outer}{inner})",
            outer = self.outerboundary.as_ref().unwrap().to_string(),
            inner = inner,
        )
    }
}

impl Geometry for Polygon { }

#[derive(Debug)]
pub struct LinearRing {
    coordinates: Option<Coordinates>,
}

impl LinearRing {
    pub fn new() -> LinearRing {
        LinearRing {
            coordinates: None,
        }
    }

    pub fn set_coordinates(&mut self, coords: Coordinates) {
        self.coordinates = Some(coords);
    }
}

#[derive(Debug)]
pub enum DataStep {
    TableSchema(TableSchema),
    DataEntry(DataEntry),
    EntryAttr(EntryAttr),
    Coordinates(Coordinates),
    GeometryCollection(GeometryCollection),
    OuterBoundary(Boundary),
    InnerBoundary(Boundary),
    Polygon(Polygon),
    LinearRing(LinearRing),
}

impl DataStep {
    pub fn add_geometry(&mut self, geom: Box<dyn Geometry>) {
        match self {
            DataStep::GeometryCollection(g) => g.add_geometry(geom),
            DataStep::DataEntry(e) => e.set_geom(geom),
            _ => panic!(),
        }
    }

    pub fn set_outer_boundary(&mut self, ob: Boundary) {
        match self {
            DataStep::Polygon(p) => p.set_outer_boundary(ob),
            _ => panic!(),
        }
    }

    pub fn add_inner_boundary(&mut self, ib: Boundary) {
        match self {
            DataStep::Polygon(p) => p.add_inner_boundary(ib),
            _ => panic!(),
        }
    }

    pub fn set_linear_ring(&mut self, lr: LinearRing) {
        match self {
            DataStep::OuterBoundary(ob) => ob.set_linear_ring(lr),
            DataStep::InnerBoundary(ib) => ib.set_linear_ring(lr),
            x => {
                dbg!(x);
                panic!();
            }
        }
    }
}
