use super::super::device::device::DeviceValue;
use super::commenter::Commenter;
use super::creator::Creator;
use super::follower::Follower;
use crate::graph_context::{
    get_optional_gdb_node_string_val, get_required_gdb_node_string_val, GraphContext,
};
use juniper::graphql_interface;
use neo4rs::Node;
use std::str::FromStr;
use strum_macros::EnumString;

pub const USER_MODEL_NAME: &str = "User";
pub const USER_MODEL_ALIAS: &str = "user";

#[derive(Copy, Clone, Eq, PartialEq, Debug, EnumString)]
pub enum UserType {
    #[strum(serialize = "Creator")]
    Creator,
    #[strum(serialize = "Follower")]
    Follower,
    #[strum(serialize = "Commenter")]
    Commenter,
}
#[graphql_interface(for = [Creator, Follower, Commenter], context = GraphContext)]
pub trait User {
    fn id(&self) -> &str;
    fn name(&self) -> Option<String>;
    fn model_type(&self) -> &str;
    async fn devices(&self, ctx: &GraphContext) -> Vec<DeviceValue>;
}

pub fn user_gdb_node_to_gql_interface(user_node: Node) -> Option<UserValue> {
    let id: String = user_node.get("id").unwrap();
    let id_str = &id[..];
    let name = get_optional_gdb_node_string_val(&user_node, "name");
    let model_name = get_required_gdb_node_string_val(&user_node, "model_name");
    let model_name_as_enum = UserType::from_str(&model_name).unwrap();

    match model_name_as_enum {
        UserType::Creator => {
            let creator = Creator::new(id_str, name);
            return Some(creator.clone().into());
        }
        UserType::Follower => {
            let follower = Follower::new(id_str, name);
            return Some(follower.clone().into());
        }
        UserType::Commenter => {
            let commenter = Commenter::new(id_str, name);
            return Some(commenter.clone().into());
        }
    }
}
